"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsMetricFactory = void 0;
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const EcsNamespace = 'AWS/ECS';
const ApplicationELBNamespace = 'AWS/ApplicationELB';
class EcsMetricFactory {
    metricCpuUtilizationAverage(clusterName, serviceName) {
        return this
            .ecsMetric("CPUUtilization" /* CPUUtilization */, clusterName, serviceName)
            .with({ statistic: aws_cloudwatch_1.Statistic.AVERAGE });
    }
    metricMemoryUtilizationAverage(clusterName, serviceName) {
        return this
            .ecsMetric("MemoryUtilization" /* MemoryUtilization */, clusterName, serviceName)
            .with({ statistic: aws_cloudwatch_1.Statistic.AVERAGE });
    }
    ecsMetric(metric, clusterName, serviceName) {
        return new aws_cloudwatch_1.Metric({
            namespace: EcsNamespace,
            metricName: metric,
            dimensionsMap: {
                ClusterName: clusterName,
                ServiceName: serviceName,
            },
        });
    }
    metricMinHealthyHostCount(targetGroup, loadBalancer) {
        return this
            .albMetric("HealthyHostCount" /* HealthyHostCount */, targetGroup, loadBalancer)
            .with({ statistic: aws_cloudwatch_1.Statistic.MINIMUM });
    }
    metricMaxUnhealthyHostCount(targetGroup, loadBalancer) {
        return this
            .albMetric("UnHealthyHostCount" /* UnHealthyHostCount */, targetGroup, loadBalancer)
            .with({ statistic: aws_cloudwatch_1.Statistic.MAXIMUM });
    }
    metricTargetResponseTime(targetGroup, loadBalancer) {
        const baseMetric = this.albMetric("TargetResponseTime" /* TargetResponseTime */, targetGroup, loadBalancer);
        return {
            min: baseMetric.with({ statistic: aws_cloudwatch_1.Statistic.MINIMUM }),
            max: baseMetric.with({ statistic: aws_cloudwatch_1.Statistic.MAXIMUM }),
            avg: baseMetric.with({ statistic: aws_cloudwatch_1.Statistic.AVERAGE }),
        };
    }
    metricRequestCount(targetGroup, loadBalancer) {
        return this
            .albMetric("RequestCount" /* RequestCount */, targetGroup, loadBalancer)
            .with({ statistic: aws_cloudwatch_1.Statistic.SUM });
    }
    metricHttpErrorStatusCodeRate(targetGroup, loadBalancer) {
        const requests = this.metricRequestCount(targetGroup, loadBalancer);
        const errors = this.metricHttpStatusCodeCount(targetGroup, loadBalancer);
        return new aws_cloudwatch_1.MathExpression({
            expression: 'http4xx + http5xx / requests',
            usingMetrics: {
                http4xx: errors.count4XX,
                http5xx: errors.count5XX,
                requests,
            },
        });
    }
    metricHttpStatusCodeCount(targetGroup, loadBalancer) {
        return {
            count2XX: this.albMetric("HTTPCode_Target_2XX_Count" /* TARGET_2XX_COUNT */, targetGroup, loadBalancer).with({ statistic: aws_cloudwatch_1.Statistic.SUM }),
            count3XX: this.albMetric("HTTPCode_Target_3XX_Count" /* TARGET_3XX_COUNT */, targetGroup, loadBalancer).with({ statistic: aws_cloudwatch_1.Statistic.SUM }),
            count4XX: this.albMetric("HTTPCode_Target_4XX_Count" /* TARGET_4XX_COUNT */, targetGroup, loadBalancer).with({ statistic: aws_cloudwatch_1.Statistic.SUM }),
            count5XX: this.albMetric("HTTPCode_Target_5XX_Count" /* TARGET_5XX_COUNT */, targetGroup, loadBalancer).with({ statistic: aws_cloudwatch_1.Statistic.SUM }),
        };
    }
    albMetric(metric, targetGroup, loadBalancer) {
        return new aws_cloudwatch_1.Metric({
            namespace: ApplicationELBNamespace,
            metricName: metric,
            dimensionsMap: {
                TargetGroup: targetGroup,
                LoadBalancer: loadBalancer,
            },
        });
    }
}
exports.EcsMetricFactory = EcsMetricFactory;
//# sourceMappingURL=data:application/json;base64,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