"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WatchRdsAurora = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const metrics_1 = require("./monitoring/aws/rds/metrics");
/**
 * @stability stable
 */
class WatchRdsAurora extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.watchful = props.watchful;
        this.cluster = props.cluster;
        this.metrics = new metrics_1.RdsAuroraMetricFactory();
        this.watchful.addSection(props.title, {
            links: [
                { title: 'AWS RDS Cluster', url: linkForRDS(this.cluster) },
            ],
        });
        const { cpuUtilizationMetric, cpuUtilizationAlarm } = this.createCpuUtilizationMonitor(props.cpuMaximumThresholdPercent);
        const { dbConnectionsMetric, dbConnectionsAlarm } = this.createDbConnectionsMonitor(props.dbConnectionsMaximumThreshold);
        const { dbReplicaLagMetric, dbReplicaLagAlarm } = this.createDbReplicaLagMonitor(props.dbReplicaLagMaximumThreshold);
        const { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm } = this.createDbBufferCacheMonitor(props.dbBufferCacheMinimumThreshold);
        const { dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric } = this.createDbDmlThroughputMonitor(props.dbThroughputMaximumThreshold);
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: `CPUUtilization/${cpuUtilizationMetric.period.toMinutes()}min`,
            width: 6,
            left: [cpuUtilizationMetric],
            leftAnnotations: [cpuUtilizationAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Connections/${dbConnectionsMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbConnectionsMetric],
            leftAnnotations: [dbConnectionsAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB Replica Lag/${dbReplicaLagMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbReplicaLagMetric],
            leftAnnotations: [dbReplicaLagAlarm.toAnnotation()],
        }), new cloudwatch.GraphWidget({
            title: `DB BufferCache Hit Ratio/${dbBufferCacheHitRatioMetric.period.toMinutes()}min`,
            width: 6,
            left: [dbBufferCacheHitRatioMetric],
            leftAnnotations: [dbBufferCacheHitRatioAlarm.toAnnotation()],
        }));
        this.watchful.addWidgets(new cloudwatch.GraphWidget({
            title: 'RDS DML Overview',
            width: 24,
            left: [dbInsertThroughputMetric, dbUpdateThroughputMetric, dbSelectThroughputMetric, dbDeleteThroughputMetric],
        }));
    }
    createCpuUtilizationMonitor(cpuMaximumThresholdPercent = 80) {
        const cpuUtilizationMetric = this.metrics.metricCpuUtilization(this.cluster.clusterIdentifier);
        const cpuUtilizationAlarm = cpuUtilizationMetric.createAlarm(this, 'CpuUtilizationAlarm', {
            alarmDescription: 'cpuUtilizationAlarm',
            threshold: cpuMaximumThresholdPercent,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { cpuUtilizationMetric, cpuUtilizationAlarm };
    }
    createDbConnectionsMonitor(dbConnectionsMaximumThreshold = 0) {
        const dbConnectionsMetric = this.metrics.metricDbConnections(this.cluster.clusterIdentifier);
        const dbConnectionsAlarm = dbConnectionsMetric.createAlarm(this, 'DbConnectionsAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbConnectionsMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbConnectionsMetric, dbConnectionsAlarm };
    }
    createDbReplicaLagMonitor(dbReplicaLagMaximumThreshold = 0) {
        const dbReplicaLagMetric = this.metrics.metricReplicaLag(this.cluster.clusterIdentifier);
        const dbReplicaLagAlarm = dbReplicaLagMetric.createAlarm(this, 'DbReplicaLagAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbReplicaLagMaximumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbReplicaLagMetric, dbReplicaLagAlarm };
    }
    createDbBufferCacheMonitor(dbBufferCacheMinimumThreshold = 0) {
        const dbBufferCacheHitRatioMetric = this.metrics.metricBufferCacheHitRatio(this.cluster.clusterIdentifier);
        const dbBufferCacheHitRatioAlarm = dbBufferCacheHitRatioMetric.createAlarm(this, 'DbBufferCacheHitRatioAlarm', {
            alarmDescription: 'dbConnectionsAlarm',
            threshold: dbBufferCacheMinimumThreshold,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 3,
        });
        return { dbBufferCacheHitRatioMetric, dbBufferCacheHitRatioAlarm };
    }
    createDbDmlThroughputMonitor(dbThroughputMaximumThreshold = 0) {
        // @ts-ignore
        const AlarmThreshold = dbThroughputMaximumThreshold;
        return this.metrics.metricDmlThroughput(this.cluster.clusterIdentifier);
    }
}
exports.WatchRdsAurora = WatchRdsAurora;
_a = JSII_RTTI_SYMBOL_1;
WatchRdsAurora[_a] = { fqn: "cdk-watchful.WatchRdsAurora", version: "0.6.29" };
function linkForRDS(cluster) {
    return `https://console.aws.amazon.com/rds/home?region=${cluster.stack.region}#database:id=${cluster.clusterIdentifier};is-cluster=true`;
}
//# sourceMappingURL=data:application/json;base64,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