#!/usr/bin/env python3
"""
Test error handling improvements for AgentListInspectorWidget

This script tests the widget with error boundaries and development mode
to get better error information when clicking on agents.
"""

import sys
import os

# Add the parent directory to the path so we can import the agent
sys.path.insert(0, os.path.join(os.path.dirname(__file__), ".."))


def test_error_handling():
    """Test the widget with comprehensive error handling."""
    print("Testing AgentListInspectorWidget with error handling...")

    try:
        from edsl.agents import Agent, AgentList
        from edsl.widgets import AgentListInspectorWidget

        # Create test agents that might cause issues
        agents = [
            Agent(
                name="Agent with Complex Data",
                traits={
                    "nested_data": {"level1": {"level2": ["item1", "item2"]}},
                    "large_list": list(range(100)),
                    "mixed_types": [1, "string", {"key": "value"}, None],
                    "unicode_text": "Test with émojis 🚀 and ünïcödé",
                    "boolean_value": True,
                    "none_value": None,
                },
                codebook={
                    "nested_data": "Complex nested data structure",
                    "large_list": "Large list of numbers",
                    "mixed_types": "List with mixed data types",
                    "unicode_text": "Text with unicode characters",
                    "boolean_value": "Boolean true/false value",
                    "none_value": "Null/None value",
                },
                instruction="I handle complex data structures and edge cases.",
            ),
            Agent(
                name="Normal Agent",
                traits={"role": "standard", "type": "normal"},
                instruction="I'm a normal agent for comparison.",
            ),
        ]

        agent_list = AgentList(agents)
        widget = AgentListInspectorWidget(agent_list)

        print(f"✓ Widget created successfully with {len(widget.agents_data)} agents")

        # Test data extraction for each agent
        for i, agent_data in enumerate(widget.agents_data):
            print(f"Agent {i}: {agent_data.get('name', 'Unknown')}")
            print(f"  - Traits: {len(agent_data.get('traits', {}))}")
            print(
                f"  - Complex data types handled: {any(isinstance(v, (dict, list)) for v in agent_data.get('traits', {}).values())}"
            )

        # Test widget methods
        summary = widget.export_summary()
        print(f"✓ Summary generated: {summary}")

        # Test individual agent access
        for i in range(len(widget.agents_data)):
            agent_data = widget.get_agent_by_index(i)
            if agent_data:
                print(f"✓ Agent {i} data accessible")
            else:
                print(f"❌ Agent {i} data not accessible")

        print("✓ Error handling test completed successfully")
        print("\n🎯 Ready for Jupyter testing:")
        print("- Development mode enabled (better error messages)")
        print("- Error boundaries added (graceful error handling)")
        print("- Complex data structures supported")
        print("- Click handlers protected with try-catch")

        return widget

    except Exception as e:
        print(f"❌ Error handling test failed: {e}")
        import traceback

        traceback.print_exc()
        return None


def create_problematic_widget():
    """Create a widget that might expose edge cases."""
    print("\nTesting edge cases that might cause React errors...")

    try:
        from edsl.agents import Agent, AgentList
        from edsl.widgets import AgentListInspectorWidget

        # Create agents with potential problem data
        problematic_agents = []

        # Agent with empty/minimal data
        problematic_agents.append(Agent())

        # Agent with very long strings
        problematic_agents.append(
            Agent(
                name="Agent with Very Long Name and Data " * 10,
                traits={
                    "very_long_text": "This is an extremely long piece of text " * 100,
                    "empty_string": "",
                    "whitespace_only": "   \n\t   ",
                    "special_chars": "!@#$%^&*()[]{}|\\:;\"'<>?,./",
                },
            )
        )

        # Agent with circular-reference-like data (as much as possible)
        recursive_data = {"self_ref": "points to self"}
        recursive_data["nested"] = (
            recursive_data.copy()
        )  # Avoid actual circular reference

        problematic_agents.append(
            Agent(
                name="Edge Case Agent",
                traits={
                    "recursive_like": recursive_data,
                    "very_nested": {"a": {"b": {"c": {"d": {"e": "deep"}}}}},
                    "number_types": [
                        1,
                        1.5,
                        0,
                        -1,
                        float("inf"),
                        float("-inf"),
                    ],  # Skip NaN as it can cause JSON issues
                },
            )
        )

        agent_list = AgentList(problematic_agents)
        widget = AgentListInspectorWidget(agent_list)

        print(f"✓ Problematic widget created with {len(widget.agents_data)} agents")

        # Test that all agents were processed
        for i, agent_data in enumerate(widget.agents_data):
            name = agent_data.get("name") or f"Agent {i+1}"
            print(f"  - Agent {i}: {name[:50]}{'...' if len(name) > 50 else ''}")

        print("✓ Edge case testing completed")
        return widget

    except Exception as e:
        print(f"❌ Edge case test failed: {e}")
        import traceback

        traceback.print_exc()
        return None


if __name__ == "__main__":
    print("Testing Enhanced Error Handling")
    print("=" * 50)

    # Test 1: Normal error handling
    widget1 = test_error_handling()

    # Test 2: Edge cases
    widget2 = create_problematic_widget()

    if widget1 and widget2:
        print(f"\n{'='*60}")
        print("Error Handling Enhancements Complete!")
        print(f"{'='*60}")
        print("\nImprovements made:")
        print("✅ Development mode builds (unminified, better errors)")
        print("✅ Error boundaries around components")
        print("✅ Try-catch blocks around event handlers")
        print("✅ Better error messages and fallbacks")
        print("✅ Comprehensive error logging")
        print("\n💡 If you still see React error #300:")
        print("- Check browser console for detailed error messages")
        print("- The error boundary should now catch and display issues")
        print("- Component should gracefully handle edge cases")
    else:
        print("\n❌ Some tests failed. Check output for details.")
