"""
Convex Space Manager - Core functionality for managing space availability in Convex.
"""

import json
from convex import ConvexClient


class ConvexSpaceManager:
    """
    A simple class to manage space availability in Convex.
    """
    
    def __init__(self, deployment_url: str):
        """
        Initialize the Convex client.
        
        Args:
            deployment_url: Your Convex deployment URL (e.g., "https://modest-pig-521.convex.cloud")
        """
        self.client = ConvexClient(deployment_url)
    
    def update_space(self, space_name: str, is_full: bool):
        """
        Update the availability status of a single space.
        
        Args:
            space_name: Name of the space (e.g., "space1", "room_a", etc.)
            is_full: True if space is full/occupied, False if available
        """
        try:
            self.client.mutation("spaces:update_fullness", {
                "spaceName": space_name,
                "isFull": is_full
            })
            print(f"✓ Updated {space_name}: {'Full' if is_full else 'Available'}")
        except Exception as e:
            print(f"✗ Error updating {space_name}: {e}")
    
    def update_multiple_spaces(self, space_names: list[str], availability_flags: list[bool]):
        """
        Update multiple spaces at once.
        
        Args:
            space_names: List of space names
            availability_flags: List of boolean flags (True = full, False = available)
        """
        if len(space_names) != len(availability_flags):
            raise ValueError("Number of space names must match number of availability flags")
        
        print(f"Updating {len(space_names)} spaces...")
        for i, (name, is_full) in enumerate(zip(space_names, availability_flags)):
            self.update_space(name, is_full)
        print("All spaces updated!")


def convex_sync(flags: list[bool], names: list[str], deployment_url: str = None):
    """
    Simple function to sync space availability - matches your original convex_synch function.
    
    Args:
        flags: List of boolean flags (True = full, False = available)
        names: List of space names
        deployment_url: Your Convex deployment URL (REQUIRED - replace the default!)
    """
    if deployment_url is None:
        raise ValueError("You must provide your Convex deployment URL! Replace 'https://your-deployment.convex.cloud' with your actual URL.")
    
    manager = ConvexSpaceManager(deployment_url)
    manager.update_multiple_spaces(names, flags)
