from dataclasses import dataclass
from typing import Collection

from ndice import d20, Dice, plus

from ..gear import Weapon, WeaponMode
from ..mechanics import Range, simplify_mods
from .damage_roll import DamageRoll, make_damage_roll
from .stats import Dex, Stren
from .talents import farsight_ranged, mighty, Talent, WeaponMastery


@dataclass(frozen=True, slots=True)
class Attack:
    weapon: Weapon
    mode: WeaponMode
    attack_roll: list[Dice]
    damage_roll: DamageRoll
    range: Range

    def __repr__(self) -> str:
        return f'<{self.__class__.__name__}: {self.weapon}, {self.mode.name}>'

    def __str__(self) -> str:
        attack_roll = ''.join([str(dice) for dice in self.attack_roll])
        range = self.range.name or 'close'
        return (
            f'{self.weapon.name}: {attack_roll} attack'
            f' | {self.damage_roll} damage'
            f' | {range.lower()} range'
        )


def make_attacks(
    weapons: Collection[Weapon],
    stren: Stren,
    dex: Dex,
    level: int,
    talents: list[Talent],
) -> list[Attack]:
    attacks = []
    for weapon in sorted(weapons, key=lambda weapon: weapon.name):
        if WeaponMode.MELEE & weapon.mode:
            attack = make_attack(
                weapon,
                WeaponMode.MELEE,
                stren,
                dex,
                level,
                talents,
            )
            attacks.append(attack)
        if WeaponMode.RANGED & weapon.mode:
            attack = make_attack(
                weapon,
                WeaponMode.RANGED,
                stren,
                dex,
                level,
                talents,
            )
            attacks.append(attack)
    return attacks


def make_attack(
    weapon: Weapon,
    mode: WeaponMode,
    stren: Stren,
    dex: Dex,
    level: int,
    talents: list[Talent],
) -> Attack:
    assert mode & weapon.mode

    attack_mods = [stren.mod if WeaponMode.MELEE == mode else dex.mod]
    damage_mods: list[Dice] = []

    for talent in talents:
        if farsight_ranged == talent and WeaponMode.RANGED == mode:
            attack_mods.append(plus(1))
        elif mighty == talent and WeaponMode.MELEE == mode:
            attack_mods.append(plus(1))
            damage_mods.append(plus(1))
        elif isinstance(talent, WeaponMastery) and weapon == talent.weapon:
            attack_mods.append(plus(1 + level // 2))
            damage_mods.append(plus(1 + level // 2))

    attack_mods = simplify_mods(*attack_mods)
    damage_mods = simplify_mods(*damage_mods)

    attack_roll = [d20] + attack_mods
    damage_roll = make_damage_roll(weapon, mode, damage_mods, level, talents)

    if WeaponMode.MELEE == mode:
        range = Range.CLOSE
    else:
        range = weapon.range & ~Range.CLOSE

    return Attack(
        weapon=weapon,
        mode=mode,
        attack_roll=attack_roll,
        damage_roll=damage_roll,
        range=range,
    )
