from __future__ import annotations

from dataclasses import dataclass
from itertools import chain

from ndice import plus

from sd.gear import Inventory, Slots, slots, Weapon
from sd.mechanics import mod_value

from .alignment import Alignment
from .ancestry import Ancestry
from .attack import Attack, make_attacks
from .background import Background
from .deity import Deity
from .gender import Gender
from .language import Language
from .profession import Profession
from .spells import Spell
from .stats import Cha, Con, Dex, Intel, Stren, Wis
from .talents import hauler, Talent, WeaponMastery
from .title import get_title


@dataclass
class Character:
    name: str
    ancestry: Ancestry
    gender: Gender
    alignment: Alignment
    background: Background
    deity: Deity

    profession: Profession | None
    level: int

    stren: Stren
    dex: Dex
    con: Con
    intel: Intel
    wis: Wis
    cha: Cha

    hp: int
    xp: int

    languages: list[Language]
    talents: list[Talent]
    spells: list[Spell]

    inventory: Inventory

    @property
    def ac(self) -> int:
        armor = self.inventory.armor()
        if armor:
            # TODO: handle multiple armor suits
            suit = armor[0]
            mods = [plus(suit.base_ac)]
            if suit.has_dex_mod:
                mods.append(self.dex.mod)
            return mod_value(*mods)
        else:
            return mod_value(plus(10), self.dex.mod)

    @property
    def gear_slots(self) -> Slots:
        value = max(10, self.stren.value)
        if hauler in self.talents:
            value += max(0, mod_value(self.con.mod))
        return value * slots

    @property
    def xp_to_level_up(self) -> int:
        return self.level * 10

    @property
    def title(self) -> str | None:
        return get_title(
            self.profession, self.alignment, self.level, self.gender
        )

    def __repr__(self) -> str:
        return f'<{self.__class__.__name__}: {self!s}>'

    def __str__(self) -> str:
        return (
            f'{self.name}, level {self.level}' + f' {self.profession.name}'
            if self.profession
            else ''
        )

    def attacks(self) -> list[Attack]:
        weapons = set(chain(self.inventory.weapons(), self.mastered_weapons()))
        return make_attacks(
            weapons, self.stren, self.dex, self.level, self.talents
        )

    def mastered_weapons(self) -> list[Weapon]:
        return [
            talent.weapon
            for talent in self.talents
            if isinstance(talent, WeaponMastery)
        ]
