from __future__ import annotations

from typing import NamedTuple, Sequence

from ndice import d6, RNG, roll, three_d6

from .cha import Cha
from .con import Con
from .dex import Dex
from .intel import Intel
from .stren import Stren
from .wis import Wis


class Stats(NamedTuple):
    stren: Stren
    dex: Dex
    con: Con
    intel: Intel
    wis: Wis
    cha: Cha

    @classmethod
    def from_rolls(cls, rolls: Sequence[int]) -> Stats:
        assert len(rolls) == 6
        return cls(
            Stren(rolls[0]),
            Dex(rolls[1]),
            Con(rolls[2]),
            Intel(rolls[3]),
            Wis(rolls[4]),
            Cha(rolls[5]),
        )

    @classmethod
    def roll_3d6_down_the_line(cls, rng: RNG) -> Stats:
        rolls = [roll(rng, three_d6) for _ in range(6)]
        return cls.from_rolls(rolls)

    @classmethod
    def roll_with_one_min_stat(cls, rng: RNG, min_stat: int) -> Stats:
        assert min_stat >= 0

        retry_count = 10
        rolls = []
        for _ in range(retry_count):
            rolls = [roll(rng, three_d6) for _ in range(6)]
            if max(rolls) >= min_stat:
                return cls.from_rolls(rolls)

        roll_to_raise = roll(rng, d6) - 1
        rolls[roll_to_raise] = min_stat
        return cls.from_rolls(rolls)
