from typing import TypeAlias

from .alignment import Alignment
from .gender import Gender
from .profession import fighter, priest, Profession, thief, wizard


Title: TypeAlias = str | dict[Gender, str]
Titles: TypeAlias = list[Title]
AlignmentTitles: TypeAlias = dict[Alignment, Titles]
ProfessionTitles: TypeAlias = dict[str, AlignmentTitles]


# Titles, SD v4.8, pp30-31

titles: ProfessionTitles = {
    'Fighter': {
        Alignment.LAWFUL: [
            'Squire',
            'Cavalier',
            'Knight',
            'Thane',
            {Gender.FEMALE: 'Lady', Gender.MALE: 'Lord'},
        ],
        Alignment.CHAOTIC: ['Knave', 'Bandit', 'Slayer', 'Reaver', 'Warlord'],
        Alignment.NEUTRAL: [
            'Warrior',
            'Barbarian',
            'Battlerager',
            'Warchief',
            'Chieftan',
        ],
    },
    'Priest': {
        Alignment.LAWFUL: [
            'Acolyte',
            'Crusader',
            'Templar',
            'Champion',
            'Paladin',
        ],
        Alignment.CHAOTIC: [
            'Initiate',
            'Zealot',
            'Cultist',
            'Scourge',
            'Chaos Knight',
        ],
        Alignment.NEUTRAL: [
            'Seeker',
            'Invoker',
            'Haruspex',
            'Mystic',
            'Oracle',
        ],
    },
    'Thief': {
        Alignment.LAWFUL: ['Footpad', 'Burglar', 'Rook', 'Underboss', 'Boss'],
        Alignment.CHAOTIC: [
            'Thug',
            'Cutthroat',
            'Shadow',
            'Assassin',
            'Wraith',
        ],
        Alignment.NEUTRAL: [
            'Robber',
            'Outlaw',
            'Rogue',
            'Renegate',
            {Gender.FEMALE: 'Bandit Queen', Gender.MALE: 'Bandit King'},
        ],
    },
    'Wizard': {
        Alignment.LAWFUL: [
            'Apprentice',
            'Conjurer',
            'Arcanist',
            'Mage',
            'Archmage',
        ],
        Alignment.CHAOTIC: [
            'Adept',
            'Channeler',
            {Gender.FEMALE: 'Witch', Gender.MALE: 'Warlock'},
            'Diabolist',
            'Sorcerer',
        ],
        Alignment.NEUTRAL: ['Shaman', 'Seer', 'Warden', 'Sage', 'Druid'],
    },
}


def get_title(
    profession: Profession | None,
    alignment: Alignment,
    level: int,
    gender: Gender,
) -> str | None:
    assert profession in (None, fighter, priest, thief, wizard)
    assert alignment in (Alignment.CHAOTIC, Alignment.LAWFUL, Alignment.NEUTRAL)
    assert 0 <= level <= 10
    assert gender in (Gender.MALE, Gender.FEMALE)

    if not profession or not level:
        return None

    index = (level - 1) // 2
    entry = titles[profession.name][alignment][index]
    return entry if isinstance(entry, str) else entry[gender]
