from dataclasses import dataclass

from ..mechanics import gp, Money
from .gear import Gear
from .slots import Slots, slots
from .unit import Unit
from .word import Word


@dataclass(frozen=True, slots=True)
class Armor(Gear):
    base_ac: int
    has_dex_mod: bool
    properties: list[str]


def make_armor(
    name: Word | str,
    unit_cost: Money,
    unit_slots: Slots,
    base_ac: int,
    *,
    has_dex_mod: bool,
    properties: list[str] | None = None,
    qty: int = 1,
) -> Armor:
    if isinstance(name, str):
        name = Word(name)
    return Armor(
        name=name,
        unit_cost=unit_cost,
        unit_slots=unit_slots,
        unit=Unit.ITEM,
        free_to_carry=0,
        qty=qty,
        base_ac=base_ac,
        has_dex_mod=has_dex_mod,
        properties=properties or [],
    )


leather_armor = make_armor(
    'leather armor', 10 * gp, 1 * slots, 11, has_dex_mod=True
)

chainmail = make_armor(
    'chainmail',
    60 * gp,
    2 * slots,
    13,
    has_dex_mod=True,
    properties=['Disadvantage on stealth', 'Disadvantage on swim'],
)

mithral_chainmail = make_armor(
    'mithral chainmail',
    4 * 60 * gp,
    1 * slots,
    13,
    has_dex_mod=True,
)

plate_mail = make_armor(
    'plate mail',
    130 * gp,
    3 * slots,
    15,
    has_dex_mod=False,
    properties=['Disadvantage on stealth', 'Unable to swim'],
)

mithral_plate_mail = make_armor(
    'mithral plate mail',
    4 * 130 * gp,
    2 * slots,
    15,
    has_dex_mod=False,
)

all_armor = [
    leather_armor,
    chainmail,
    mithral_chainmail,
    plate_mail,
    mithral_plate_mail,
]
