from __future__ import annotations

from dataclasses import dataclass, replace
from fractions import Fraction
from typing import Any

from sd.mechanics import gp, Money, sp

from .slots import Slots, slots
from .unit import Unit
from .word import Word


@dataclass(frozen=True, slots=True)
class Gear:
    name: Word
    unit_cost: Money
    unit_slots: Slots
    unit: Unit
    free_to_carry: int
    qty: int

    def __post_init__(self) -> None:
        assert isinstance(self.name, Word)
        assert self.unit_cost >= Money(cp=0)
        assert self.unit_slots >= Slots(0)
        assert self.free_to_carry >= 0
        assert self.qty > 0

    @property
    def cost(self) -> Money:
        return self.qty * self.unit_cost

    @property
    def kind(self) -> str:
        return self.name.singular

    @property
    def slots(self) -> Slots:
        net_qty = max(0, self.qty - self.free_to_carry)
        return Fraction(net_qty) * self.unit_slots

    def __add__(self, other: Any) -> Gear:
        if isinstance(other, Gear):
            assert self.kind == other.kind
            return replace(self, qty=self.qty + other.qty)
        else:
            return NotImplemented

    def __repr__(self) -> str:
        name = self.name.singular if self.name.is_regular else self.name
        qty = f', qty={self.qty!r}' if self.qty > 1 else ''
        return f'{self.__class__.__name__}({name!r}{qty})'

    def __rmul__(self, other: Any) -> Gear:
        if isinstance(other, int):
            return replace(self, qty=self.qty * other)
        else:
            return NotImplemented

    def __str__(self) -> str:
        parts = []

        if self.unit in [Unit.BAG, Unit.FLASK]:
            parts.append(f'{self.qty} {self.unit.value.inflect(self.qty)} of ')
        elif self.unit in [Unit.FOOT, Unit.ITEM] and self.qty > 1:
            tick = "'" if self.unit == Unit.FOOT else ''
            parts.append(f'{self.qty}{tick} ')

        parts.append(self.name.inflect(self.qty))

        whole_slots = round(self.slots)
        if whole_slots != 1 * slots:
            parts.append(' ')
            parts.append(f'({whole_slots} slots)')
        return ''.join(parts)

    def __sub__(self, other: Any) -> Gear:
        if isinstance(other, Gear):
            assert self.kind == other.kind
            return replace(self, qty=self.qty - other.qty)
        else:
            return NotImplemented


def make_gear(
    name: Word | str,
    unit_cost: Money,
    unit_slots: Slots = 1 * slots,
    *,
    unit: Unit = Unit.ITEM,
    free_to_carry: int = 0,
    qty: int = 1,
) -> Gear:
    if isinstance(name, str):
        name = Word(name)
    return Gear(
        name=name,
        unit_cost=unit_cost,
        unit_slots=unit_slots,
        unit=unit,
        free_to_carry=free_to_carry,
        qty=qty,
    )


# Basic Gear table, SD v4.8, p35
arrow = make_gear('arrow', 1 * gp / 20, 1 * slots / 20)
backpack = make_gear('backpack', 2 * gp, free_to_carry=1)
caltrops = make_gear(Word('caltrops', 'caltrops'), 5 * sp, unit=Unit.BAG)
crossbow_bolt = make_gear('crossbow bolt', 1 * gp / 20, 1 * slots / 20)
crowbar = make_gear('crowbar', 5 * sp)
flask = make_gear('flask', 3 * sp)
flint_and_steel = make_gear(Word('flint and steel', 'flints and steel'), 5 * sp)
gem = make_gear('gem', 0 * gp, 1 * slots / 10)
gold_piece = make_gear(
    'gold piece',
    1 * gp,
    1 * slots / 100,
    free_to_carry=100,
)
grappling_hook = make_gear('grappling hook', 1 * gp)
holy_symbol = make_gear('holy symbol', 0 * gp, 0 * slots)
iron_spike = make_gear('iron spike', 1 * gp / 10, 1 * slots / 10)
lantern = make_gear('lantern', 5 * gp)
mirror = make_gear('mirror', 10 * gp)
oil_flask = make_gear(Word('oil', 'oil'), 5 * sp, unit=Unit.FLASK)
pole = make_gear('pole', 5 * sp)
rations = make_gear(Word('rations', 'rations'), 5 * sp / 3, 1 * slots / 3)
rope = make_gear(
    Word('rope', 'rope'), 1 * gp / 60, 1 * slots / 60, unit=Unit.FOOT
)
thievery_tools = make_gear('thievery tools', 0 * gp, 0 * slots)
torch = make_gear(Word('torch', 'torches'), 5 * sp)


# Crawling Kit table, SD v4.8, p36
crawling_kit = [
    backpack,
    flint_and_steel,
    2 * torch,
    3 * rations,
    10 * iron_spike,
    grappling_hook,
    60 * rope,
]
