from collections.abc import Collection, Iterator
from typing import Any, Sequence

from .armor import Armor
from .gear import Gear
from .slots import Slots, slots
from .weapon import Weapon


type GearList = Gear | Sequence[GearList]


class Inventory(Collection[Gear]):
    def __init__(self) -> None:
        self.__gear: dict[str, Gear] = {}

    @property
    def slots(self) -> Slots:
        return round(sum((gear.slots for gear in self), start=0 * slots))

    def __contains__(self, item: Any) -> bool:
        kind = kind_of(item)
        return kind in self.__gear if kind else False

    def __getitem__(self, index: Gear | str) -> Gear:
        kind = kind_of(index)
        if kind is None:
            raise TypeError(f'Unexpected index type {type(index)}')
        return self.__gear[kind]

    def __iter__(self) -> Iterator[Gear]:
        return iter(self.__gear.values())

    def __len__(self) -> int:
        return len(self.__gear)

    def add(self, *items: GearList) -> None:
        for item in items:
            if isinstance(item, Gear):
                if item in self:
                    item = self[item] + item
                self.__gear[item.kind] = item
            elif isinstance(item, list):
                self.add(*item)
            else:
                raise TypeError(f'Unexpected item type {type(item)}')

    def armor(self) -> list[Armor]:
        return [
            gear for gear in self.__gear.values() if isinstance(gear, Armor)
        ]

    def remove(self, gear: Gear) -> None:
        del self.__gear[gear.kind]

    def weapons(self) -> list[Weapon]:
        return [
            gear for gear in self.__gear.values() if isinstance(gear, Weapon)
        ]


def kind_of(item: Gear | str) -> str | None:
    if isinstance(item, Gear):
        return item.kind
    elif isinstance(item, str):
        return item
    else:
        return None
