from dataclasses import dataclass

from ndice import d4, d6, d8, d10, d12, Dice

from ..mechanics import cp, gp, Money, Range, sp
from .gear import Gear
from .grip import Grip
from .slots import Slots, slots
from .unit import Unit
from .weapon_mode import WeaponMode
from .word import Word


@dataclass(frozen=True, slots=True)
class Weapon(Gear):
    mode: WeaponMode
    range: Range
    damage: Dice | tuple[Dice, Dice]
    grip: Grip = Grip.ONE_HANDED
    finesse: bool = False
    loading: bool = False
    thrown: bool = False

    @property
    def is_two_handed(self) -> bool:
        return self.grip == Grip.TWO_HANDED

    @property
    def is_versatile(self) -> bool:
        return self.grip == Grip.ONE_HANDED | Grip.TWO_HANDED


def make_weapon(
    name: Word | str,
    unit_cost: Money,
    damage: Dice | tuple[Dice, Dice],
    *,
    mode: WeaponMode = WeaponMode.MELEE,
    range: Range = Range.CLOSE,
    grip: Grip = Grip.ONE_HANDED,
    finesse: bool = False,
    thrown: bool = False,
    loading: bool = False,
    unit_slots: Slots = 1 * slots,
    qty: int = 1,
) -> Weapon:
    if isinstance(name, str):
        name = Word(name)
    return Weapon(
        name=name,
        unit_cost=unit_cost,
        unit_slots=unit_slots,
        unit=Unit.ITEM,
        free_to_carry=0,
        qty=qty,
        mode=mode,
        range=range,
        damage=damage,
        grip=grip,
        finesse=finesse,
        loading=loading,
        thrown=thrown,
    )


# Weapon table, SD v4.8, p37
bastard_sword = make_weapon(
    'bastard sword',
    10 * gp,
    (d8, d10),
    grip=Grip.ONE_HANDED | Grip.TWO_HANDED,
    unit_slots=2 * slots,
)

club = make_weapon('club', 5 * cp, d4)

crossbow = make_weapon(
    'crossbow',
    8 * gp,
    d6,
    mode=WeaponMode.RANGED,
    range=Range.FAR,
    grip=Grip.TWO_HANDED,
    loading=True,
)

dagger = make_weapon(
    'dagger',
    1 * gp,
    d4,
    mode=WeaponMode.MELEE | WeaponMode.RANGED,
    range=Range.CLOSE | Range.NEAR,
    finesse=True,
    thrown=True,
)

greataxe = make_weapon(
    'greataxe',
    10 * gp,
    (d8, d10),
    grip=Grip.ONE_HANDED | Grip.TWO_HANDED,
    unit_slots=2 * slots,
)

greatsword = make_weapon(
    'greatsword',
    12 * gp,
    d12,
    grip=Grip.TWO_HANDED,
    unit_slots=2 * slots,
)

javelin = make_weapon(
    'javelin',
    5 * sp,
    d4,
    mode=WeaponMode.MELEE | WeaponMode.RANGED,
    range=Range.CLOSE | Range.FAR,
    thrown=True,
)

longbow = make_weapon(
    'longbow',
    8 * gp,
    d8,
    mode=WeaponMode.RANGED,
    range=Range.FAR,
    grip=Grip.TWO_HANDED,
)

longsword = make_weapon('longsword', 9 * gp, d8)

mace = make_weapon('mace', 5 * gp, d6)

shortbow = make_weapon(
    'shortbow',
    6 * gp,
    d4,
    mode=WeaponMode.RANGED,
    range=Range.FAR,
    grip=Grip.TWO_HANDED,
)

shortsword = make_weapon('shortsword', 7 * gp, d6)

spear = make_weapon(
    'spear',
    5 * sp,
    d6,
    mode=WeaponMode.MELEE | WeaponMode.RANGED,
    range=Range.CLOSE | Range.NEAR,
    thrown=True,
)

staff = make_weapon('staff', 5 * sp, d4, grip=Grip.TWO_HANDED)

warhammer = make_weapon('warhammer', 10 * gp, d10, grip=Grip.TWO_HANDED)

all_weapons = [
    bastard_sword,
    club,
    crossbow,
    dagger,
    greataxe,
    greatsword,
    javelin,
    longbow,
    longsword,
    mace,
    shortbow,
    shortsword,
    spear,
    staff,
    warhammer,
]
