from __future__ import annotations

from dataclasses import dataclass
from fractions import Fraction
from math import ceil
from typing import Any, Iterator


@dataclass(frozen=True, order=True, slots=True)
class Money:
    cp: Fraction

    def __init__(
        self,
        /,
        gp: Fraction | int = Fraction(0),
        sp: Fraction | int = Fraction(0),
        cp: Fraction | int = Fraction(0),
    ):
        cp = 100 * Fraction(gp) + 10 * Fraction(sp) + Fraction(cp)
        object.__setattr__(self, 'cp', cp)

    @property
    def coins(self) -> tuple[int, int, int]:
        sp, cp = divmod(ceil(self.cp), 10)
        gp, sp = divmod(sp, 10)
        return gp, sp, cp

    def __add__(self, other: Any) -> Money:
        if isinstance(other, Money):
            return Money(cp=self.cp + other.cp)
        else:
            return NotImplemented

    def __int__(self) -> int:
        return int(self.cp)

    def __iter__(self) -> Iterator[int]:
        return iter(self.coins)

    def __repr__(self) -> str:
        return f'{self.__class__.__name__}(cp={self.cp})'

    def __rmul__(self, other: Any) -> Money:
        factor = other_factor(other)
        if factor is None:
            return NotImplemented
        else:
            return Money(cp=self.cp * factor)

    def __str__(self) -> str:
        gp, sp, cp = self
        parts = []
        if gp:
            parts.append(f'{gp} gp')
        if sp:
            parts.append(f'{sp} sp')
        if cp or not parts:
            parts.append(f'{cp} cp')
        return ', '.join(parts)

    def __sub__(self, other: Any) -> Money:
        if isinstance(other, Money):
            return Money(cp=self.cp - other.cp)
        else:
            return NotImplemented

    def __truediv__(self, other: Any) -> Money:
        factor = other_factor(other)
        if factor is None:
            return NotImplemented
        else:
            return Money(cp=self.cp / factor)


gp = Money(gp=1)
sp = Money(sp=1)
cp = Money(cp=1)


def other_factor(other: Any) -> Fraction | None:
    if isinstance(other, Fraction):
        return other
    elif isinstance(other, int):
        return Fraction(other)
    else:
        return None
