from __future__ import annotations

import sys

from argparse import ArgumentParser
from dataclasses import dataclass
from pathlib import Path

from ndice import high, low, mid, PRNG, RNG, rng


@dataclass
class Options:
    max_hp_at_first_level: bool
    reroll_if_no_stat_is_14_plus: bool
    rng: RNG
    starting_level: int

    @classmethod
    def parse(cls, args: list[str] | None = None) -> Options:
        if args is None:
            arg_zero, *args = sys.argv
            prog = find_program_name(arg_zero)
        else:
            prog = 'sd'
            args = args

        parser = ArgumentParser(prog=prog)
        parser.add_argument(
            '-0',
            '--zero',
            action='store_const',
            const=0,
            default=1,
            dest='starting_level',
            help='generate a level-0 character',
        )
        parser.add_argument(
            '--max-hp-at-first-level',
            action='store_true',
            default=False,
            help='get max HP at first level instead of rolling',
        )
        parser.add_argument(
            '--reroll-if-no-stat-is-14-plus',
            action='store_true',
            default=False,
            help='if no stat is 14+, reroll',
        )
        parser.add_argument(
            '-r',
            '--rng',
            default=rng,
            type=rng_type,
            help='random, low, mid, high or integer seed; default is "random"',
        )
        namespace = parser.parse_args(args)
        return Options(
            max_hp_at_first_level=namespace.max_hp_at_first_level,
            reroll_if_no_stat_is_14_plus=namespace.reroll_if_no_stat_is_14_plus,
            rng=namespace.rng,
            starting_level=namespace.starting_level,
        )


def find_program_name(arg_zero: str) -> str:
    path = Path(arg_zero)
    if '__main__' == path.stem:
        return path.parent.name
    else:
        return path.stem


def rng_from_seed(value: str) -> RNG:
    try:
        seed = int(value)
        return PRNG(seed)
    except ValueError:
        raise ValueError(f'Invalid rng seed value: "{value}"')


def rng_type(value: str) -> RNG:
    match value:
        case 'random':
            return rng
        case 'low':
            return low
        case 'mid':
            return mid
        case 'high':
            return high
        case _:
            return rng_from_seed(value)
