from typing import cast

from ndice import RNG

from ..characters import Character, farsight, farsight_ranged, farsight_spell
from ..characters import priest, Talent, TalentChoice, weapon_mastery
from ..characters import WeaponMastery, wizard
from ..gear import crossbow
from ..tables import Table


def choose_talent(
    rng: RNG, character: Character, talent_choice: TalentChoice
) -> Talent:
    if farsight == talent_choice:
        return choose_farsight(rng, character)
    elif weapon_mastery == talent_choice:
        return choose_weapon_mastery(rng, character)
    else:
        choices_table = Table.with_values(*talent_choice.choices)
        return choices_table.roll(rng)


def first_talent_choice(talents: list[Talent]) -> TalentChoice | None:
    return next(
        (talent for talent in talents if isinstance(talent, TalentChoice)),
        None,
    )


def choose_talents(rng: RNG, character: Character) -> None:
    # talent choices may be affected by a character's existing talents,
    # so remove and resolve talent choices one at a time
    while talent_choice := first_talent_choice(character.talents):
        character.talents.remove(talent_choice)
        talent = choose_talent(rng, character, talent_choice)
        character.talents.append(talent)


# See ruling "Farsight for random characters"
def choose_farsight(rng: RNG, character: Character) -> Talent:
    farsight_table = Table.with_values(*farsight.choices)
    if 0 == character.level:
        return farsight_table.roll(rng)
    else:
        if priest == character.profession:
            if crossbow in character.inventory:
                return farsight_table.roll(rng)
            else:
                return farsight_spell
        elif wizard == character.profession:
            return farsight_spell
        else:
            return farsight_ranged


# See ruling "Weapon mastery for random first level character"
def choose_weapon_mastery(rng: RNG, character: Character) -> Talent:
    previous_masteries = [
        talent
        for talent in character.talents
        if isinstance(talent, WeaponMastery)
    ]
    previously_mastered_weapons = [
        previous_mastery.weapon for previous_mastery in previous_masteries
    ]

    weapon_masteries = cast(list[WeaponMastery], weapon_mastery.choices)
    masteries = [
        mastery
        for mastery in weapon_masteries
        if mastery not in previous_masteries
    ]
    weapons = [
        weapon
        for weapon in character.inventory.weapons()
        if weapon not in previously_mastered_weapons
    ]

    if 0 == len(weapons):
        masteries_table = Table.with_values(*masteries)
        return masteries_table.roll(rng)

    if 1 == len(weapons):
        weapon = weapons.pop()
    else:
        weapons_table = Table.with_values(*weapons)
        weapon = weapons_table.roll(rng)

    choice = [mastery for mastery in masteries if mastery.weapon == weapon]
    assert 1 == len(choice)
    return choice[0]
