from ..characters import Attack, Character, Language, Profession, Stat
from ..gear import WeaponMode


class CharacterSheet:
    def __init__(self, character: Character):
        self.character = character

    def render(self) -> str:
        c = self.character
        title = c.title or ''
        alignment = c.alignment.title()
        melee_attacks, ranged_attacks = split_melee_and_ranged_attacks(
            c.attacks()
        )

        lines = [
            f'{title} {c.name}',
            f'{alignment} {c.ancestry}',
            level_line(c.level, c.profession, c.xp, c.xp_to_level_up),
            '',
            f'Background: {c.background.name}',
            f'- {c.background.description}',
            f'Deity: {c.deity}',
            f'- {c.deity.description}',
        ]

        lines.extend(language_lines(c.languages))

        lines.extend(
            [
                '',
                f'{stat(c.stren)}  {stat(c.intel)}',
                f'{stat(c.dex)}  {stat(c.wis)}',
                f'{stat(c.con)}  {stat(c.cha)}',
                '',
                f'HP: {c.hp:3}       AC: {c.ac:3}',
            ]
        )

        if melee_attacks:
            lines.extend(['', 'Melee Attacks:'])
            lines.extend([f'- {attack}' for attack in melee_attacks])
        if ranged_attacks:
            lines.extend(['', 'Ranged Attacks:'])
            lines.extend([f'- {attack}' for attack in ranged_attacks])

        lines.extend(['', 'Talents/Spells:'])
        lines.extend([f'- {talent}' for talent in c.talents])
        lines.extend([f'- {spell}' for spell in c.spells])

        lines.extend(
            [
                '',
                f'Gear: {c.inventory.slots}/{c.gear_slots} slots',
            ]
        )
        lines.extend([f'- {item}' for item in c.inventory])

        return '\n'.join([line.strip() for line in lines])


def split_melee_and_ranged_attacks(
    attacks: list[Attack],
) -> tuple[list[Attack], list[Attack]]:
    melee = []
    ranged = []
    for attack in attacks:
        if WeaponMode.MELEE == attack.mode:
            melee.append(attack)
        else:
            ranged.append(attack)
    return melee, ranged


def language_lines(languages: list[Language]) -> list[str]:
    line = 'Languages:'
    language_list = ', '.join(str(language) for language in languages)
    if len(languages) > 3:
        return [line, '- ' + language_list]
    else:
        return [line + ' ' + language_list]


def level_line(
    level: int, profession: Profession | None, xp: int, xp_to_level_up: int
) -> str:
    line = f'Level {level}'
    if level:
        return line + f' {profession} (xp: {xp}/{xp_to_level_up})'
    else:
        return line


def stat(stat: Stat) -> str:
    mod = f'({stat.mod})' if stat.mod.number else '    '
    return f'{stat.abbreviation.upper()}: {stat.value:2} {mod}'
