import gzip
import sys
from contextlib import AbstractContextManager
from pathlib import Path

if sys.version_info >= (3, 11):
    from typing import Self
else:
    from typing_extensions import Self

if sys.version_info >= (3, 12):
    from typing import override
else:
    from typing_extensions import override

import os
from types import TracebackType

from ._parsing import ParsedFile


class FoamFileIO(AbstractContextManager["FoamFileIO"]):
    def __init__(self, path: os.PathLike[str] | str) -> None:
        self.path = Path(path).absolute()

        self.__parsed: ParsedFile | None = None
        self.__missing: bool | None = None
        self.__defer_io = 0

    @override
    def __enter__(self) -> Self:
        """Read the file from disk if not already read, and defer writing of changes until the context is exited."""
        if self.__defer_io == 0:
            self._get_parsed(missing_ok=True)
        self.__defer_io += 1
        return self

    @override
    def __exit__(
        self,
        exc_type: type[BaseException] | None,
        exc_val: BaseException | None,
        exc_tb: TracebackType | None,
    ) -> None:
        """If this is the outermost context, write any deferred file changes to disk."""
        self.__defer_io -= 1
        if self.__defer_io == 0:
            assert self.__parsed is not None
            if self.__parsed.modified:
                contents = self.__parsed.contents

                if self.path.suffix == ".gz":
                    contents = gzip.compress(contents)

                self.path.write_bytes(contents)
                self.__parsed.modified = False
                self.__missing = False

    def _get_parsed(self, *, missing_ok: bool = False) -> ParsedFile:
        if not self.__defer_io:
            try:
                contents = self.path.read_bytes()
            except FileNotFoundError:
                self.__missing = True
                contents = b""
            else:
                self.__missing = False
                if self.path.suffix == ".gz":
                    contents = gzip.decompress(contents)

            if self.__parsed is None or self.__parsed.contents != contents:
                self.__parsed = ParsedFile(contents)

        assert self.__parsed is not None
        assert self.__missing is not None

        if self.__missing and not self.__parsed.modified and not missing_ok:
            raise FileNotFoundError(self.path)

        return self.__parsed

    @override
    def __repr__(self) -> str:
        return f"{type(self).__qualname__}('{self.path}')"
