import numpy as np
import pytest

from copious.cv.graphics import SplineInterpolator, calc_polyline_length, lerp, are_points_within_boundary


def test_lerp():
    res = lerp((1, 2, 3), (4, 5, 6), [0.  , 0.25, 0.5 , 0.75, 1. ])
    np.testing.assert_almost_equal(res, np.array([[1.,   2.,   3.  ],
                                                [1.75, 2.75, 3.75],
                                                [2.5 , 3.5 , 4.5 ],
                                                [3.25, 4.25, 5.25],
                                                [4.  , 5.  , 6.  ]]))
    

def test_calc_polyline_length():
    points = [
        [0, 0, 0], 
        [1, 1, 0],
        [2, 1, 1],
        [2, 2, 2]
    ]
    length = calc_polyline_length(points)
    assert length == pytest.approx(4.242640687)

def test_calc_polyline_length_2():
    points = np.array([[-38.74764008,  20.37718027,  -3.04      ],
       [-39.65805492,  23.27317982,  -3.04      ]])
    length = calc_polyline_length(points)
    assert length == pytest.approx(3.03573196)



def test_spline_interpolator():
    x = np.array([1.50000000e+00,  1.46779335e+00,  1.37398960e+00,  1.22676039e+00,
        1.03883011e+00,  8.26229207e-01,  6.06729930e-01,  3.98123593e-01,
        2.16513689e-01,  7.47974314e-02, -1.85116241e-02, -6.00025687e-02,
       -5.15578176e-02, -1.11022302e-16,  8.37644653e-02,  1.86012966e-01,
        2.91751238e-01,  3.86372455e-01,  4.57257104e-01,  4.95146477e-01,
        4.95146477e-01,  4.57257104e-01,  3.86372455e-01,  2.91751238e-01,
        1.86012966e-01,  8.37644653e-02,  2.22044605e-16, -5.15578176e-02,
       -6.00025687e-02, -1.85116241e-02,  7.47974314e-02,  2.16513689e-01,
        3.98123593e-01,  6.06729930e-01,  8.26229207e-01,  1.03883011e+00,
        1.22676039e+00,  1.37398960e+00,  1.46779335e+00,  1.50000000e+00])
    y = np.array([0.00000000e+00,  2.38539637e-01,  4.58705129e-01,  6.43853519e-01,
        7.80630188e-01,  8.60195723e-01,  8.79000054e-01,  8.39027698e-01,
        7.47491969e-01,  6.16012269e-01,  4.59361215e-01,  2.93912021e-01,
        1.35946792e-01,  1.92296269e-16, -1.02592845e-01, -1.64793108e-01,
       -1.84492304e-01, -1.64617919e-01, -1.12703754e-01, -3.99723561e-02,
        3.99723561e-02,  1.12703754e-01,  1.64617919e-01,  1.84492304e-01,
        1.64793108e-01,  1.02592845e-01,  3.84592537e-16, -1.35946792e-01,
       -2.93912021e-01, -4.59361215e-01, -6.16012269e-01, -7.47491969e-01,
       -8.39027698e-01, -8.79000054e-01, -8.60195723e-01, -7.80630188e-01,
       -6.43853519e-01, -4.58705129e-01, -2.38539637e-01, -3.67394040e-16])
    
    interpolator = SplineInterpolator([x, y], s=0, k=3)
    
    u = [0.        , 0.03616231, 0.07211603, 0.10765441, 0.14257429,
       0.17667804, 0.20977543, 0.24168569, 0.27223968, 0.30128236,
       0.3286756 , 0.35430164, 0.37806752, 0.39991104, 0.41980905,
       0.43778945, 0.45394843, 0.46847409, 0.48167407, 0.49399473,
       0.50600527, 0.51832593, 0.53152591, 0.54605157, 0.56221055,
       0.58019095, 0.60008896, 0.62193248, 0.64569836, 0.6713244 ,
       0.69871764, 0.72776032, 0.75831431, 0.79022457, 0.82332196,
       0.85742571, 0.89234559, 0.92788397, 0.96383769, 1.        ]    

    interpolated = interpolator.interpolate(u)
    np.testing.assert_almost_equal(interpolated[0], np.array([ 1.50000000e+00,  1.46779335e+00,  1.37398960e+00,  1.22676039e+00,
        1.03883011e+00,  8.26229207e-01,  6.06729930e-01,  3.98123593e-01,
        2.16513689e-01,  7.47974314e-02, -1.85116241e-02, -6.00025687e-02,
       -5.15578176e-02, -1.16226473e-16,  8.37644653e-02,  1.86012966e-01,
        2.91751238e-01,  3.86372455e-01,  4.57257104e-01,  4.95146477e-01,
        4.95146477e-01,  4.57257104e-01,  3.86372455e-01,  2.91751238e-01,
        1.86012966e-01,  8.37644653e-02,  2.34187669e-16, -5.15578176e-02,
       -6.00025687e-02, -1.85116241e-02,  7.47974314e-02,  2.16513689e-01,
        3.98123593e-01,  6.06729930e-01,  8.26229207e-01,  1.03883011e+00,
        1.22676039e+00,  1.37398960e+00,  1.46779335e+00,  1.50000000e+00]))
    np.testing.assert_almost_equal(interpolated[1], np.array([-2.58233385e-18,  2.38539637e-01,  4.58705129e-01,  6.43853519e-01,
        7.80630188e-01,  8.60195723e-01,  8.79000054e-01,  8.39027698e-01,
        7.47491969e-01,  6.16012269e-01,  4.59361215e-01,  2.93912021e-01,
        1.35946792e-01,  1.94289029e-16, -1.02592845e-01, -1.64793108e-01,
       -1.84492304e-01, -1.64617919e-01, -1.12703754e-01, -3.99723561e-02,
        3.99723561e-02,  1.12703754e-01,  1.64617919e-01,  1.84492304e-01,
        1.64793108e-01,  1.02592845e-01,  4.05925293e-16, -1.35946792e-01,
       -2.93912021e-01, -4.59361215e-01, -6.16012269e-01, -7.47491969e-01,
       -8.39027698e-01, -8.79000054e-01, -8.60195723e-01, -7.80630188e-01,
       -6.43853519e-01, -4.58705129e-01, -2.38539637e-01, -3.67394040e-16]))


def test_points_within_boundary():
    boundary_points = np.array([[0, 0], [0, 5], [5, 0], [5, 5]])
    points_to_check = np.array([[1, 1], [2, 2], [3, 3], [4, 4]])
    assert are_points_within_boundary(points_to_check, boundary_points)


def test_points_outside_boundary():
    boundary_points = np.array([[0, 0], [0, 5], [5, 0], [5, 5]])
    points_to_check = np.array([[6, 6], [7, 7], [-1, -1], [-2, -2]])
    assert not are_points_within_boundary(points_to_check, boundary_points)


def test_boundary_as_single_point():
    boundary_points = np.array([[2, 2]])
    points_to_check = np.array([[2, 2], [2, 2], [2, 2]])
    assert are_points_within_boundary(points_to_check, boundary_points)