import time
import httpx
import asyncio
import datetime
from itertools import count
from collections import deque
from typing import Any, Dict, Deque, Tuple, Union, Optional, AsyncGenerator, Callable, Literal, AsyncIterable
from aioquic.quic.events import QuicEvent, StreamDataReceived, StreamReset, ConnectionTerminated
from aioquic.quic.configuration import QuicConfiguration
from aioquic.asyncio.client import connect
from aioquic.asyncio.protocol import QuicConnectionProtocol



class GNExceptions:
    class ConnectionError:
        class openconnector():
            """Ошибка подключения к серверу openconnector.gn"""
            
            class connection(Exception):
                def __init__(self, message="Ошибка подключения к серверу openconnector.gn. Сервер не найден."):
                    super().__init__(message)

            class timeout(Exception):
                def __init__(self, message="Ошибка подключения к серверу openconnector.gn. Проблема с сетью или сервер перегружен."):
                    super().__init__(message)

            class data(Exception):
                def __init__(self, message="Ошибка подключения к серверу openconnector.gn. Сервер не подтвердил подключение."):
                    super().__init__(message)

        class dns():
            """Ошибка подключения к серверу dns.core"""
            class connection(Exception):
                def __init__(self, message="Ошибка подключения к серверу dns.core Сервер не найден."):
                    super().__init__(message)

            class timeout(Exception):
                def __init__(self, message="Ошибка подключения к серверу dns.core Проблема с сетью или сервер перегружен"):
                    super().__init__(message)
    
            class data(Exception):
                def __init__(self, message="Ошибка подключения к серверу dns.core Сервер не подтвердил подключение."):
                    super().__init__(message)

        class connector():
            """Ошибка подключения к серверу <?>~connector.gn"""
            
            class connection(Exception):
                def __init__(self, message="Ошибка подключения к серверу <?>~connector.gn. Сервер не найден."):
                    super().__init__(message)

            class timeout(Exception):
                def __init__(self, message="Ошибка подключения к серверу <?>~connector.gn. Проблема с сетью или сервер перегружен"):
                    super().__init__(message)
    
            class data(Exception):
                def __init__(self, message="Ошибка подключения к серверу <?>~connector.gn. Сервер не подтвердил подключение."):
                    super().__init__(message)

        class client():
            """Ошибка клиента"""
            
            class connection(Exception):
                def __init__(self, message="Ошибка подключения к серверу. Сервер не найден."):
                    super().__init__(message)

            class timeout(Exception):
                def __init__(self, message="Ошибка подключения к серверу. Проблема с сетью или сервер перегружен"):
                    super().__init__(message)
    
            class data(Exception):
                def __init__(self, message="Ошибка подключения к серверу. Сервер не подтвердил подключение."):
                    super().__init__(message)
                    

from KeyisBTools import TTLDict
from KeyisBTools.cryptography.sign import S1
from gnobjects.net.objects import GNRequest, GNResponse, Url

from ._crt import crt_client


s1 = S1()


async def chain_async(first_item, rest: AsyncIterable) -> AsyncGenerator:
    yield first_item
    async for x in rest:
        yield x



class AsyncClient:
    def __init__(self):
        self.__dns_core__ipv4 = '51.250.85.38:52943'
        self.__dns_gn__ipv4: Optional[str] = None

        self.__current_session = {}
        self.__request_callbacks = {}
        self.__response_callbacks = {}

        self._active_connections: Dict[str, QuicClient] = {}

        self.__dns_client: Optional[AsyncClient] = None
        self._dns_cache: TTLDict = TTLDict()

    async def getCoreDNS(self, domain: str, token: Optional[bytes] = None, use_cache: bool = True, keep_alive: bool = False) -> str:
        if use_cache:
            resuilt = self._dns_cache.get(domain)
            if resuilt is not None:
                return resuilt

        if ':' in domain and domain.split('.')[-1].split(':')[0].isdigit() and domain.split(':')[-1].isdigit():
            return domain

        try:
            if self.__dns_client is None:
                self.__dns_client = AsyncClient()

            if token is not None:
                payload = {'sign': {'alg': 'KeyisB-c-s-m1', 'data': s1.sign(token)}}
            else:
                payload = None

            r1 = await self.__dns_client.request(GNRequest('GET', Url(f'gn://{self.__dns_core__ipv4}/getIp?d={domain}'), payload=payload), keep_alive=keep_alive)

            if r1.command != 'ok':
                raise GNExceptions.ConnectionError.dns.data

            if r1.payload is None:
                raise GNExceptions.ConnectionError.dns.data

            r1_data = r1.payload

            result = r1_data['ip'] + ':' + str(r1_data['port']) # type: ignore

            self._dns_cache.set(domain, result, r1_data.get('ttl')) # type: ignore

            return result

        except httpx.TimeoutException:
            raise GNExceptions.ConnectionError.dns.timeout
        except:
            raise GNExceptions.ConnectionError.dns.connection

    async def getGNDNS(self, domain: str, token: Optional[bytes] = None, use_cache: bool = True, keep_alive: bool = False) -> str:
        if use_cache:
            resuilt = self._dns_cache.get(domain)
            if resuilt is not None:
                return resuilt

        if ':' in domain and domain.split('.')[-1].split(':')[0].isdigit() and domain.split(':')[-1].isdigit():
            return domain

        if self.__dns_gn__ipv4 is None:
            self.__dns_gn__ipv4 = await self.getCoreDNS('dns.gn')

        try:
            if self.__dns_client is None:
                self.__dns_client = AsyncClient()
            
            if token is not None:
                payload = {'sign': {'alg': 'KeyisB-c-s-m1', 'data': s1.sign(token)}}
            else:
                payload = None

            r1 = await self.__dns_client.request(GNRequest('GET', Url(f'gn://{self.__dns_gn__ipv4}/getIp?d={domain}'), payload=payload), keep_alive=keep_alive)

            if r1.command != 'ok':
                raise GNExceptions.ConnectionError.dns.data

            
            if r1.payload is None:
                raise GNExceptions.ConnectionError.dns.data
            
            r1_data = r1.payload

            result = r1_data['ip'] + ':' + str(r1_data['port']) # type: ignore

            self._dns_cache.set(domain, result, r1_data.get('ttl')) # type: ignore

            return result

        except httpx.TimeoutException:
            raise GNExceptions.ConnectionError.dns.timeout
        except:
            raise GNExceptions.ConnectionError.dns.connection

    async def getDNS(self, domain: str, token: Optional[bytes] = None, use_cache: bool = True, keep_alive: bool = False):
        if domain.endswith(('.core', '.gw', '.gn', '.cdn', '.sys', '.gwis', '.abs')):
            return await self.getCoreDNS(domain=domain, use_cache=use_cache, keep_alive=keep_alive)
        else:
            return await self.getGNDNS(domain=domain, token=token, use_cache=use_cache, keep_alive=keep_alive)

    def addRequestCallback(self, callback: Callable, name: str):
        self.__request_callbacks[name] = callback

    def addResponseCallback(self, callback: Callable, name: str):
        self.__response_callbacks[name] = callback

  
    async def connect(self, domain: str, restart_connection: bool = False, reconnect_wait: float = 10, keep_alive: bool = True) -> 'QuicClient':
        print('Запрос подключения')
        if not restart_connection and domain in self._active_connections:
            c = self._active_connections[domain]
            print(f'Подключение уже было [{c.status}]')
            if c.status == 'connecting':
                print('ждем поделючения')
                try:
                    await asyncio.wait_for(c.connect_future, reconnect_wait)
                    print('дождались')
                    if c.status == 'active':
                        return c
                    elif c.status == 'connecting': # если очень долго подключаемся, то кидаем ошибку
                        await self.disconnect(domain)
                        raise GNExceptions.ConnectionError.client.timeout
                    elif c.status == 'disconnect':
                        raise GNExceptions.ConnectionError.client.connection
                except:
                    print('Заново соеденяемся...')
                    await self.disconnect(domain)
                        
            else:
                return c

        c = QuicClient()
        c.status = 'connecting'
        self._active_connections[domain] = c

        data = await self.getDNS(domain)

        data = data.split(':')



        def f(domain):
            self._active_connections.pop(domain)

        c._disconnect_signal = f # type: ignore
        c._domain = domain # type: ignore

        await c.connect(data[0], int(data[1]), keep_alive=keep_alive)
        await c.connect_future

        return c

    async def disconnect(self, domain):
        if domain not in self._active_connections:
            return
        
        await self._active_connections[domain].disconnect()


    def _return_token(self, bigToken: str, s: bool = True) -> str:
        return bigToken[:128] if s else bigToken[128:]


    async def request(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]], keep_alive: bool = True, restart_connection: bool = False, reconnect_wait: float = 10) -> GNResponse:
        if isinstance(request, GNRequest):
            

            c = await self.connect(request.url.hostname, restart_connection, reconnect_wait, keep_alive=keep_alive)
                


            for f in self.__request_callbacks.values():
                asyncio.create_task(f(request))

            r = await c.asyncRequest(request)

            for f in self.__response_callbacks.values():
                asyncio.create_task(f(r))

            return r
        
        else:
            c: Optional[QuicClient] = None

            async def wrapped(request) -> AsyncGenerator[GNRequest, None]:
                async for req in request:
                    if req.gn_protocol is None:
                        req.setGNProtocol(self.__current_session['protocols'][0])
                    req._stream = True

                    for f in self.__request_callbacks.values():
                        asyncio.create_task(f(req))

                    nonlocal c
                    if c is None:  # инициализируем при первом req
                        c = await self.connect(request.url.hostname, restart_connection, reconnect_wait, keep_alive=keep_alive)

                    yield req

            gen = wrapped(request)
            first_req = await gen.__anext__()

            if c is None:
                raise GNExceptions.ConnectionError.client.data

            r = await c.asyncRequest(chain_async(first_req, gen))

            for f in self.__response_callbacks.values():
                asyncio.create_task(f(r))

            return r


    # async def requestStream(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]]) -> AsyncGenerator[GNResponse, None]:
    #     """
    #     Build and send a async request.
    #     """
    #     if isinstance(request, GNRequest):
    #         if request.gn_protocol is None:
    #             request.setGNProtocol(self.__current_session['protocols'][0])
                
    #         for f in self.__request_callbacks.values():
    #             asyncio.create_task(f(request))

    #         async for response in self.client.asyncRequestStream(request):
                    
    #             for f in self.__response_callbacks.values():
    #                 asyncio.create_task(f(response))

    #             yield response
    #     else:
    #         async def wrapped(request) -> AsyncGenerator[GNRequest, None]:
    #             async for req in request:
    #                 if req.gn_protocol is None:
    #                     req.setGNProtocol(self.__current_session['protocols'][0])
                            
    #                 for f in self.__request_callbacks.values():
    #                     asyncio.create_task(f(req))
                        
    #                 req._stream = True
    #                 yield req
    #         async for response in self.client.asyncRequestStream(wrapped(request)):
                
    #             for f in self.__response_callbacks.values():
    #                 asyncio.create_task(f(response))

    #             yield response

class RawQuicClient(QuicConnectionProtocol):

    SYS_RATIO_NUM = 9  # SYS 9/10
    SYS_RATIO_DEN = 10
    KEEPALIVE_INTERVAL = 10  # сек
    KEEPALIVE_IDLE_TRIGGER = 30  # сек

    # ────────────────────────────────────────────────────────────────── init ─┐
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.quicClient: QuicClient = None # type: ignore

        self._sys_stream_id: int = None # type: ignore
        self._queue_sys: Deque[Tuple[int, bytes, bool]] = deque()
        self._queue_user: Deque[Tuple[int, bytes, bool]] = deque()

        # <‑‑ Future | Queue[bytes | None]
        self._inflight: Dict[int, Union[asyncio.Future, asyncio.Queue[Optional[GNResponse]]]] = {}
        self._inflight_streams: Dict[int, bytearray] = {}
        self._sys_inflight: Dict[int, asyncio.Future] = {}
        self._buffer: Dict[Union[int, str], bytearray] = {}

        self._sys_budget = self.SYS_RATIO_NUM
        self._sys_id_gen = count(1)  # int64 message‑id generator

        self._last_activity = time.time()
        self._running = True
        self._ping_id_gen = count(1)  # int64 ping‑id generator

    # ───────────────────────────────────────── private helpers ─┤
    def _activity(self):
        self._last_activity = time.time()

    async def _keepalive_loop(self):
        while self._running:
            await asyncio.sleep(self.KEEPALIVE_INTERVAL)
            idle_time = time.time() - self._last_activity
            if idle_time > self.KEEPALIVE_IDLE_TRIGGER:
                self._quic.send_ping(next(self._ping_id_gen))
                self.transmit()
                self._last_activity = time.time()

    def stop(self):
        self._running = False

    # ───────────────────────────────────────────── events ─┤
    def quic_event_received(self, event: QuicEvent) -> None:  # noqa: C901
        # ─── DATA ───────────────────────────────────────────
        if isinstance(event, StreamDataReceived):
            #print(event)
            # SYS поток
            if event.stream_id == self._sys_stream_id:
                buf = self._buffer.setdefault("sys", bytearray())
                buf.extend(event.data)
                while True:
                    if len(buf) < 12:
                        break
                    msg_id = int.from_bytes(buf[:8], "little")
                    size = int.from_bytes(buf[8:12], "little")
                    if len(buf) < 12 + size:
                        break
                    payload = bytes(buf[12 : 12 + size])
                    del buf[: 12 + size]
                    fut = self._sys_inflight.pop(msg_id, None) if msg_id else None
                    if fut and not fut.done():
                        fut.set_result(payload)
            # USER поток
            else:
                handler = self._inflight.get(event.stream_id)
                if handler is None:
                    return
                
                # Чтение в зависимости от режима
                if isinstance(handler, asyncio.Queue): # стрим от сервера
                    # получаем байты

                    buf = self._buffer.setdefault(event.stream_id, bytearray())
                    buf.extend(event.data)

                    if len(buf) < 8: # не дошел даже frame пакета
                        return
                    
                    # получаем длинну пакета
                    mode, stream, lenght = GNResponse.type(buf)

                    if mode != 4: # не наш пакет
                        self._buffer.pop(event.stream_id)
                        return
                    
                    if not stream: # клиент просил стрим, а сервер прислал один пакет
                        self._buffer.pop(event.stream_id)
                        return
                    
                    # читаем пакет
                    if len(buf) < lenght: # если пакет не весь пришел, пропускаем
                        return
                    
                    # пакет пришел весь

                    # берем пакет
                    data = buf[:lenght]

                    # удаляем его из буфера
                    del buf[:lenght]
                        
                    
                    r = GNResponse.deserialize(data, 2)
                    handler.put_nowait(r)
                    if event.end_stream:
                        handler.put_nowait(None)
                        self._buffer.pop(event.stream_id)
                        self._inflight.pop(event.stream_id, None)



                else:  # Future
                    buf = self._buffer.setdefault(event.stream_id, bytearray())
                    buf.extend(event.data)
                    if event.end_stream:
                        self._inflight.pop(event.stream_id, None)
                        data = bytes(self._buffer.pop(event.stream_id, b""))
                        if not handler.done():
                            handler.set_result(data)

        # ─── RESET ──────────────────────────────────────────
        elif isinstance(event, StreamReset):
            handler = self._inflight.pop(event.stream_id, None) or self._sys_inflight.pop(
                event.stream_id, None
            )
            if handler is None:
                return
            if isinstance(handler, asyncio.Queue):
                handler.put_nowait(None)
            else:
                if not handler.done():
                    handler.set_exception(RuntimeError("stream reset"))


        elif isinstance(event, ConnectionTerminated):
            print("QUIC connection closed")
            print("Error code:", event.error_code)
            print("Reason:", event.reason_phrase)
            if self.quicClient is None:
                return
            
            asyncio.create_task(self.quicClient.disconnect())


    # ─────────────────────────────────────────── scheduler ─┤
    def _enqueue(self, sid: int, blob: bytes, end_stream: bool, is_sys: bool):
        (self._queue_sys if is_sys else self._queue_user).append((sid, blob, end_stream))

    def _schedule_flush(self):
        while (self._queue_sys or self._queue_user) and self._quic._close_event is None:
            q = None
            if self._queue_sys and (self._sys_budget > 0 or not self._queue_user):
                q = self._queue_sys
                self._sys_budget -= 1
            elif self._queue_user:
                q = self._queue_user
                self._sys_budget = self.SYS_RATIO_NUM
            if q is None:
                break
            sid, blob, end_stream = q.popleft()
            print(f'Отправка стрима {sid}')
            self._quic.send_stream_data(sid, blob, end_stream=end_stream)
        self.transmit()
        self._activity()

    # ─────────────────────────────────────────── public API ─┤
    async def ensure_sys_stream(self):
        if self._sys_stream_id is None:
            self._sys_stream_id = self._quic.get_next_available_stream_id()
            self._enqueue(self._sys_stream_id, b"", False, True)  # dummy
            self._schedule_flush()

    async def send_sys(self, request: GNRequest, response: bool = False) -> Optional[bytes]:
        await self.ensure_sys_stream()
        if response:
            msg_id = next(self._sys_id_gen)
            blob = request.serialize(2)
            payload = (
                msg_id.to_bytes(8, "little") + len(blob).to_bytes(4, "little") + blob
            )
            fut = asyncio.get_running_loop().create_future()
            self._sys_inflight[msg_id] = fut
            self._enqueue(self._sys_stream_id, payload, False, True)
            self._schedule_flush()
            return await fut
        payload = (0).to_bytes(8, "little") + request.serialize(2)
        self._enqueue(self._sys_stream_id, payload, False, True)
        self._schedule_flush()
        return None

    async def request(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]]):
        if isinstance(request, GNRequest):
            blob = request.serialize(2)
            sid = self._quic.get_next_available_stream_id()
            self._enqueue(sid, blob, True, False)
            self._schedule_flush()

            
            fut = asyncio.get_running_loop().create_future()
            self._inflight[sid] = fut
            return await fut
        
        else:
            sid = self._quic.get_next_available_stream_id()
            #if sid in self._quic._streams and not self._quic._streams[sid].is_finished:

            async def _stream_sender(sid, request: AsyncGenerator[GNRequest, Any]):
                _last = None
                async for req in request:
                    _last = req
                    blob = req.serialize(2)
                    self._enqueue(sid, blob, False, False)


                    self._schedule_flush()

                    print(f'Отправлен stream запрос {req}')
                

                _last.setPayload(None)
                _last.setMethod('gn:end-stream')
                blob = _last.serialize(2)
                self._enqueue(sid, blob, True, False)
                self._schedule_flush()
            
            asyncio.create_task(_stream_sender(sid, request))

                
            fut = asyncio.get_running_loop().create_future()
            self._inflight[sid] = fut
            return await fut
    
    async def requestStream(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]]) -> asyncio.Queue[GNResponse]:
        if isinstance(request, GNRequest):
            blob = request.serialize(2)
            sid = self._quic.get_next_available_stream_id()
            self._enqueue(sid, blob, False, False)
            self._schedule_flush()

            
            q = asyncio.Queue()
            self._inflight[sid] = q
            return q
            
        else:
            sid = self._quic.get_next_available_stream_id()

            async def _stream_sender(sid, request: AsyncGenerator[GNRequest, Any]):
                _last = None
                async for req in request:
                    _last = req
                    blob = req.serialize(2)
                    self._enqueue(sid, blob, False, False)


                    self._schedule_flush()

                    print(f'Отправлен stream запрос {req}')
                

                _last.setPayload(None)
                _last.setMethod('gn:end-stream')
                blob = _last.serialize(2)
                self._enqueue(sid, blob, True, False)
                self._schedule_flush()
            
            asyncio.create_task(_stream_sender(sid, request))

                
            q = asyncio.Queue()
            self._inflight[sid] = q
            return q

        

class QuicClient:
    """Обёртка‑фасад над RawQuicClient."""

    def __init__(self):
        self._quik_core: Optional[RawQuicClient] = None
        self._client_cm = None
        self._disconnect_signal = None
        self._domain = None

        self.status: Literal['active', 'connecting', 'disconnect']

        self.connect_future = asyncio.get_event_loop().create_future()
        self.connection_time: datetime.datetime = None # type: ignore

    async def connect(self, ip: str, port: int, keep_alive: bool = True):
        self.status = 'connecting'
        self.connection_time = datetime.datetime.now()
        cfg = QuicConfiguration(is_client=True, alpn_protocols=["gn:backend"])
        cfg.load_verify_locations(cadata=crt_client)
        cfg.idle_timeout = 40

        self._client_cm = connect(
            ip,
            port,
            configuration=cfg,
            create_protocol=RawQuicClient,
            wait_connected=True,
        )


        try:
            self._quik_core = await self._client_cm.__aenter__() # type: ignore
            self._quik_core.quicClient = self

            if keep_alive:
                asyncio.create_task(self._quik_core._keepalive_loop())

            self.status = 'active'
            if not self.connect_future.done():
                self.connect_future.set_result(True)
        except Exception as e:
            print(f'Error connecting: {e}')
            if not self.connect_future.done():
                self.connect_future.set_exception(GNExceptions.ConnectionError.client.connection)
            await self._client_cm.__aexit__(None, None, None)

    async def disconnect(self):
        self.status = 'disconnect'
        
        if self._quik_core is not None:
            self._quik_core.stop()
        

        if self._disconnect_signal is not None:
            self._disconnect_signal(self._domain)
        

        if self._quik_core is not None:


            for fut in self._quik_core._inflight.values():
                if isinstance(fut, asyncio.Queue):
                    del fut
                else:
                    fut.set_exception(Exception)

            for fut in self._quik_core._sys_inflight.values():
                fut.set_exception(Exception)


            self._quik_core.close()
            await self._quik_core.wait_closed()
            self._quik_core = None

    async def asyncRequest(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]]) -> GNResponse:
        if self._quik_core is None:
            raise RuntimeError("Not connected")
        
        resp = await self._quik_core.request(request)
        return GNResponse.deserialize(resp, 2)

    async def asyncRequestStream(self, request: Union[GNRequest, AsyncGenerator[GNRequest, Any]]) -> AsyncGenerator[GNResponse, None]:
        
        if self._quik_core is None:
            raise RuntimeError("Not connected")

        queue = await self._quik_core.requestStream(request)

        while True:
            chunk = await queue.get()
            if chunk is None or chunk.command == 'gn:end-stream':
                break
            yield chunk


