from typing import Any

import orjson

from velithon.responses import HTMLResponse

swagger_ui_default_parameters = {
    'dom_id': '#swagger-ui',
    'layout': 'BaseLayout',
    'deepLinking': True,
    'showExtensions': True,
    'showCommonExtensions': True,
}


def get_swagger_ui_html(
    *,
    openapi_url: str,
    title: str,
    swagger_js_url: str = 'https://cdn.jsdelivr.net/npm/swagger-ui-dist@5/swagger-ui-bundle.js',
    swagger_css_url: str = 'https://cdn.jsdelivr.net/npm/swagger-ui-dist@5/swagger-ui.css',
    swagger_favicon_url: str = 'https://res.cloudinary.com/dslpmba3s/image/upload/v1746254848/logo_wgobg2.svg',
    oauth2_redirect_url: str | None = None,
    init_oauth: dict[str, Any] | None = None,
) -> HTMLResponse:
    html = f"""
    <!DOCTYPE html>
    <html>
    <head>
    <link type="text/css" rel="stylesheet" href="{swagger_css_url}">
    <link rel="shortcut icon" href="{swagger_favicon_url}">
    <title>{title}</title>
    </head>
    <body>
    <div id="swagger-ui">
    </div>
    <script src="{swagger_js_url}"></script>
    <!-- `SwaggerUIBundle` is now available on the page -->
    <script>
    const ui = SwaggerUIBundle({{
        url: '{openapi_url}',
    """
    for key, value in swagger_ui_default_parameters.items():
        html += f'{orjson.dumps(key).decode()}: {orjson.dumps(value).decode()},\n'

    if oauth2_redirect_url:
        html += f"oauth2RedirectUrl: window.location.origin + '{oauth2_redirect_url}',"

    html += """
    presets: [
        SwaggerUIBundle.presets.apis,
        SwaggerUIBundle.SwaggerUIStandalonePreset
        ],
    })"""

    if init_oauth:
        html += f"""
        ui.initOAuth({orjson.dumps(init_oauth).decode()})
        """

    html += """
    </script>
    </body>
    </html>
    """
    return HTMLResponse(html)
