import base64
import json
import logging
import math
import os
import re
from textwrap import indent
from typing import Any, Dict, List, NamedTuple, Optional, Tuple, TypeVar, Union

import requests
import yaml
from markdown import markdown
from pydantic import BaseModel, Field

from multiqc import config, report
from multiqc.core.log_and_rich import run_with_spinner
from multiqc.plots.plot import Plot
from multiqc.types import Anchor, SampleName
from multiqc.utils import config_schema

logger = logging.getLogger(__name__)

# List of known reasoning models
REASONING_MODELS = {
    # OpenAI reasoning models
    "o1",
    "o1-preview",
    "o1-mini",
    "o3",
    "o3-mini",
    "o3-pro",
    "o4-mini",
    # Anthropic Claude 4 series (extended thinking models)
    "claude-3-7-sonnet-latest",
    "claude-sonnet-4-0",
    "claude-haiku-4-0",
    "claude-opus-4-0",
}


def is_reasoning_model(model_name: str) -> bool:
    """Check if a model is a reasoning model based on its name."""
    if not model_name:
        return False

    # Check if model name starts with any of the reasoning model prefixes
    model_lower = model_name.lower()
    return any(model_lower.startswith(prefix) for prefix in REASONING_MODELS)


_MULTIQC_DESCRIPTION = """\
You are an expert in bioinformatics, sequencing technologies, genomics data analysis, and adjacent fields.

You are given findings from a MultiQC report, generated by a bioinformatics workflow.
MultiQC supports various bioinformatics tools that output QC metrics, and aggregates those metrics
into a single report. It outputs a "General Statistics" table with key metrics for each sample across
all tools. That table is followed by more detailed sections from specific tools, that can include tables,
as well as plots of different types (bar plot, line plot, scatter plot, heatmap, etc.)
"""

_PROMPT_BASE = (
    _MULTIQC_DESCRIPTION
    + """
You are given data from such a report. Your task is to analyse the data, and
give 1-2 bullet points of a very short and concise overall summary for the results.
Don't waste words: mention only the important QC issues. If there are no issues, just say so.
Just print one or two bullet points, nothing else.
Please do not add any extra headers to the response.

Use markdown to format your reponse for readability. Use directives with pre-defined classes
.text-green, .text-red, and .text-yellow to highlight severity, e.g. :span[39.2%]{.text-red}.
Highlight any mentioned sample names or sample named prefixes or suffixes with a sample directive,
and make sure to use the same color classes for severity, e.g. :sample[A1001.2003]{.text-yellow}
or :sample[A1001]{.text-yellow}. Do not put multiple sample names inside one directive.

You must use only multiples of 4 spaces to indent nested lists.
"""
)

_SHORT_EXAMPLE = """
Two examples of short summaries:

- :span[11/13 samples]{.text-green} show consistent metrics within expected ranges.
- :sample[A1001.2003]{.text-red} and :sample[A1001.2004]{.text-red} exhibit extremely high percentage of :span[duplicates]{.text-red} (:span[65.54%]{.text-red} and :span[83.14%]{.text-red}, respectively).

- All samples show good quality metrics with :span[75.7-77.0%]{.text-green} CpG methylation and :span[76.3-86.0%]{.text-green} alignment rates
- :sample[2wk]{.text-yellow} samples show slightly higher duplication (:span[11-15%]{.text-yellow}) compared to :sample[1wk]{.text-green} samples (:span[6-9%]{.text-green})'
"""

_PROMPT_FULL_DETAILED_SUMMARY_INSTRUCTIONS = """
Next, generate a more detailed summary of the results, and your recommendations for the next steps.

Make sure to use a multiple of 4 spaces to indent nested lists.

Highlight severity of the values and the sample names with directives and the pre-defined classes, similar to the short summary,
as in the example below.
"""

_EXAMPLE_SUMMARY_FOR_FULL = """\
- :span[11/13 samples]{.text-green} show consistent metrics within expected ranges.
- :sample[A1001.2003]{.text-red} and :sample[A1001.2004]{.text-red} exhibit extremely high percentage of :span[duplicates]{.text-red} (:span[65.54%]{.text-red} and :span[83.14%]{.text-red}, respectively).
"""

_PROMPT_EXAMPLE_SUMMARY_FOR_FULL = f"""\
Example, formatted as YAML of 2 sections (summary, detailed_analysis):

summary: |
{indent(_EXAMPLE_SUMMARY_FOR_FULL, "    ")}
"""

_EXAMPLE_DETAILED_SUMMARY = """\
**Analysis**

- :sample[A1002]{.text-yellow} and :sample[A1003]{.text-yellow} groups (:span[11/13 samples]{.text-green}) show good quality metrics, with consistent GC content (38-39%), read lengths (125 bp), and acceptable levels of duplicates and valid pairs.
- :sample[A1001.2003]{.text-red} and :sample[A1001.2004]{.text-red} show severe quality issues:
    - Extremely high duplicate rates (:span[65.54%]{.text-red} and :span[83.14%]{.text-red})
    - Low percentages of valid pairs (:span[37.2%]{.text-red} and :span[39.2%]{.text-red})
    - High percentages of failed modules in FastQC (:span[33.33%]{.text-red})
    - Significantly higher total sequence counts (:span[141.9M]{.text-red} and :span[178.0M]{.text-red}) compared to other samples
    - FastQC results indicate that :sample[A1001.2003]{.text-red} and :sample[A1001.2004]{.text-red} have a slight :span[GC content]{.text-red} bias at 39.5% against most other samples having 38.0%, which indicates a potential contamination that could be the source of other anomalies in quality metrics.

- :sample[A1002-1007]{.text-yellow} shows some quality concerns:
    - Low percentage of valid pairs (:span[48.08%]{.text-yellow})
    - Low percentage of passed Di-Tags (:span[22.51%]{.text-yellow})

- Overrepresented sequences analysis reveals adapter contamination in several samples, particularly in :sample[A1001.2003]{.text-yellow} (up to :span[35.82%]{.text-yellow} in Read 1).
- HiCUP analysis shows that most samples have acceptable levels of valid pairs, with :sample[A1003]{.text-green} group generally performing better than :sample[A1002]{.text-yellow} group.

**Recommendations**

- Remove :sample[A1001.2003]{.text-red} and :sample[A1200.2004]{.text-red} from further analysis due to severe quality issues.
- Investigate the cause of low valid pairs and passed Di-Tags in :sample[A1002-1007]{.text-yellow}. Consider removing it if the issue cannot be resolved.
- Perform adapter trimming on all samples, particularly focusing on :sample[A1001]{.text-red} group.
- Re-run the Hi-C analysis pipeline after removing problematic samples and performing adapter trimming.
- Investigate the cause of higher duplication rates in :sample[A1002]{.text-yellow} group compared to :sample[A1003]{.text-green} group, although they are still within acceptable ranges.
- Consider adjusting the Hi-C protocol or library preparation steps to improve the percentage of valid pairs, especially for :sample[A1002]{.text-yellow} group.
"""

_PROMPT_EXAMPLE_DETAILED_SUMMARY = f"""\
detailed_analysis: |
{indent(_EXAMPLE_DETAILED_SUMMARY, "    ")}
"""

PROMPT_SHORT = f"""\
{_PROMPT_BASE}

{_SHORT_EXAMPLE}
"""

PROMPT_FULL = f"""\
{_PROMPT_BASE}

{_PROMPT_FULL_DETAILED_SUMMARY_INSTRUCTIONS}

{_PROMPT_EXAMPLE_SUMMARY_FOR_FULL}

{_PROMPT_EXAMPLE_DETAILED_SUMMARY}
"""


class InterpretationOutput(BaseModel):
    summary: str = Field(description="A very short and concise overall summary")
    detailed_analysis: Optional[str] = Field(description="Detailed analysis", default=None)

    def markdown_to_html(self, text: str) -> str:
        """
        Convert markdown to HTML
        """
        # First convert pseudonyms back to original names if needed
        text = deanonymize_sample_names(text)

        html = markdown(text)
        # Find and replace directives :span[1.23%]{.text-red} -> <span..., handle multiple matches in one string
        html = re.sub(
            r":span\[([^\]]+?)\]\{\.text-(green|red|yellow)\}",
            r"<span class='text-\2'>\1</span>",
            html,
        )
        # similarly, find and replace directives :sample[A1001.2003]{.text-red} -> <sample...
        html = re.sub(
            r":sample\[([^\]]+?)\]\{\.text-(green|red|yellow)\}",
            r"<sample data-toggle='tooltip' title='Click to highlight in the report' class='text-\2'>\1</sample>",
            html,
        )
        return html

    # def format_text(self) -> str:
    #     """
    #     Format to markdown to display in Seqera AI
    #     """
    #     summary = deanonymize_sample_names(self.summary)
    #     detailed = deanonymize_sample_names(self.detailed_analysis) if self.detailed_analysis else None
    #     return f"## Analysis\n{summary}" + (f"\n\n{detailed}" if detailed else "")


class InterpretationResponse(BaseModel):
    interpretation: InterpretationOutput
    model: str
    thread_id: Optional[str] = None


ResponseT = TypeVar("ResponseT")


class Client:
    def __init__(self, api_key: Optional[str] = None):
        self.name: str
        self.title: str
        self.model: str
        self.api_key: Optional[str] = api_key
        self.prompt_short = config.ai_prompt_short or PROMPT_SHORT
        self.prompt_full = config.ai_prompt_full or PROMPT_FULL

    def _query(self, prompt: str):
        raise NotImplementedError

    def interpret_report_short(self, report_content: str) -> InterpretationResponse:
        response = self._query(self.prompt_short + "\n\n" + report_content)

        return InterpretationResponse(
            interpretation=InterpretationOutput(summary=response.content),
            model=response.model,
        )

    def interpret_report_full(self, report_content: str) -> InterpretationResponse:
        response = self._query(self.prompt_full + "\n\n" + report_content)

        try:
            output = yaml.safe_load(response.content)
            return InterpretationResponse(
                interpretation=InterpretationOutput(**output),
                model=response.model,
            )
        except Exception:
            return InterpretationResponse(
                interpretation=InterpretationOutput(summary=response.content),
                model=response.model,
            )

    def max_tokens(self) -> int:
        raise NotImplementedError

    FALLBACK_LOG_PRINTED = False

    def n_tokens(self, text: str) -> int:
        """
        Estimate token count. Using here tiktoken library. It's an OpenAI tokenizer, so it's not ideal for Anthropic,
        but better than nothing. Anthropic's tokenizer is only available through API and counts towards the API quota :'(
        """
        try:
            import tiktoken

            model = self.model if self.name == "openai" else "gpt-4o"
            encoding = tiktoken.encoding_for_model(model)
            return len(encoding.encode(text))
        except Exception as e:
            if not self.FALLBACK_LOG_PRINTED:
                logger.warning(f"Fail to call tiktoken, falling back to rough token estimation. Error: {e}")
                self.FALLBACK_LOG_PRINTED = True
            return int(len(text) / 1.5)

    def _request_with_error_handling_and_retries(
        self, url: str, headers: Dict[str, Any], body: Dict[str, Any], retries: Optional[int] = None
    ) -> Dict[str, Any]:
        """Make a request with retries and exponential backoff.

        Args:
            send_request: Callable that makes the actual request
            retries: Number of retries (0 means no retries)
        """
        import time

        retries = retries or config.ai_retries or 3
        attempt = 0
        while True:
            try:
                if config.verbose:
                    response = requests.post(url, headers=headers, json=body)
                else:
                    response = run_with_spinner(
                        "ai",
                        f"Summarizing report with {self.title}...",
                        lambda: requests.post(url, headers=headers, json=body),
                    )
                response.raise_for_status()
                return response.json()

            except Exception as e:
                attempt += 1
                if attempt > retries:
                    logger.error(f"Failed to get a response from {self.title}. Error: {e}")
                    if config.verbose:
                        # redact content in messages
                        for m in body["messages"]:
                            if isinstance(m["content"], list):
                                for chunk in m["content"]:
                                    chunk["text"] = "<redacted>"
                            elif isinstance(m["content"], str):
                                m["content"] = "<redacted>"
                        logger.debug(f"Request body: {json.dumps(body, indent=2)}")
                    raise

                # Calculate backoff time - start at 1s and double each retry
                backoff = 2 ** (attempt - 1)
                logger.warning(
                    f"Failed to get a response from {self.title} (attempt {attempt}/{retries}). Error: {e}. "
                    f"Retrying in {backoff}s..."
                )
                time.sleep(backoff)


class OpenAiClient(Client):
    def __init__(self, api_key: str, endpoint: Optional[str] = None):
        super().__init__(api_key)

        if endpoint:
            self.endpoint = endpoint
            if not config.ai_model:
                raise ValueError("Custom OpenAI endpoint is set, but no model is provided. Please set config.ai_model")
            self.model = config.ai_model
            self.name = "custom"
            self.title = endpoint
        else:
            self.endpoint = "https://api.openai.com/v1/chat/completions"
            self.model = config.ai_model or "gpt-4o"
            self.name = "openai"
            self.title = "OpenAI"

    def max_tokens(self) -> int:
        if config.ai_custom_context_window:
            return config.ai_custom_context_window

        # Reasoning models have different context windows
        if is_reasoning_model(self.model):
            if self.model.startswith("o1-preview") or self.model.startswith("o1-mini"):
                return 128000  # o1-preview and o1-mini have 128k context
            elif (
                self.model.startswith("claude-opus-4")
                or self.model.startswith("claude-sonnet-4")
                or self.model.startswith("claude-haiku-4")
            ):
                return 200000  # Claude 4 series have 200k context
            else:
                return 200000  # Other reasoning models (o3, o4-mini) have 200k context

        return 128000  # Default for regular models

    class ApiResponse(NamedTuple):
        content: str
        model: str

    def _query(self, prompt: str, extra_options: Optional[Dict[str, Any]] = None) -> ApiResponse:
        is_reasoning = is_reasoning_model(self.model)

        if is_reasoning:
            logger.debug(f"Using reasoning model: {self.model}")

        body: Dict[str, Any] = {}

        # For reasoning models, don't include temperature and other unsupported parameters
        if not is_reasoning:
            body["temperature"] = 0.0

        if config.ai_extra_query_options:
            body.update(config.ai_extra_query_options)
        if extra_options:
            body.update(extra_options)

        # Set up messages - reasoning models benefit from developer messages
        messages = []
        if is_reasoning:
            # Add developer message to encourage proper formatting for reasoning models
            messages.append(
                {
                    "role": "developer",
                    "content": "You are a helpful assistant expert in bioinformatics. Please provide clear, well-formatted responses using markdown where appropriate.",
                }
            )

        messages.append({"role": "user", "content": prompt})

        body.update(
            {
                "model": self.model,
                "messages": messages,
            }
        )

        # For OpenAI reasoning models, use max_completion_tokens and reasoning effort
        # Note: Claude 4 series are also reasoning models but may use different parameters
        if is_reasoning and not (
            self.model.startswith("claude-opus-4")
            or self.model.startswith("claude-sonnet-4")
            or self.model.startswith("claude-haiku-4")
        ):
            # Use config or reasonable default for max_completion_tokens
            max_completion_tokens = config.ai_max_completion_tokens or 4000
            if config.ai_extra_query_options and "max_completion_tokens" in config.ai_extra_query_options:
                max_completion_tokens = config.ai_extra_query_options["max_completion_tokens"]

            body["max_completion_tokens"] = max_completion_tokens

            # Add reasoning effort parameter - use config or default to medium
            reasoning_effort = config.ai_reasoning_effort or "medium"
            if config.ai_extra_query_options and "reasoning_effort" in config.ai_extra_query_options:
                reasoning_effort = config.ai_extra_query_options["reasoning_effort"]

            body["reasoning_effort"] = reasoning_effort
            logger.debug(
                f"OpenAI reasoning model parameters: max_completion_tokens={max_completion_tokens}, reasoning_effort={reasoning_effort}"
            )
        elif is_reasoning and (
            self.model.startswith("claude-opus-4")
            or self.model.startswith("claude-sonnet-4")
            or self.model.startswith("claude-haiku-4")
        ):
            # Claude 4 reasoning models - use extended thinking if enabled
            if config.ai_extended_thinking:
                thinking_budget_tokens = config.ai_thinking_budget_tokens or 10000
                if config.ai_extra_query_options and "thinking_budget_tokens" in config.ai_extra_query_options:
                    thinking_budget_tokens = config.ai_extra_query_options["thinking_budget_tokens"]

                body["thinking"] = {"type": "enabled", "budget_tokens": thinking_budget_tokens}
                logger.debug(f"Claude 4 model with extended thinking enabled: budget_tokens={thinking_budget_tokens}")
            else:
                logger.debug("Claude 4 model without extended thinking (ai_extended_thinking=False)")

        if config.ai_auth_type == "api-key":
            headers = {
                "Content-Type": "application/json",
                "api-key": self.api_key,
            }
        else:
            headers = {
                "Content-Type": "application/json",
                "Authorization": f"Bearer {self.api_key}",
            }
        response = self._request_with_error_handling_and_retries(
            self.endpoint,
            headers=headers,
            body=body,
        )
        return OpenAiClient.ApiResponse(
            content=response["choices"][0]["message"]["content"],
            model=response.get("model", self.model),
        )


class AnthropicClient(Client):
    def __init__(self, api_key: str):
        super().__init__(api_key)
        self.model = (
            config.ai_model if config.ai_model and config.ai_model.startswith("claude") else "claude-sonnet-4-0"
        )
        self.name = "anthropic"
        self.title = "Anthropic"

    def max_tokens(self) -> int:
        return 200000

    class ApiResponse(NamedTuple):
        content: str
        model: str

    def _query(self, prompt: str) -> ApiResponse:
        response = self._request_with_error_handling_and_retries(
            "https://api.anthropic.com/v1/messages",
            headers={
                "Content-Type": "application/json",
                "x-api-key": self.api_key,
                "anthropic-version": "2023-06-01",
            },
            body={
                "model": self.model,
                "max_tokens": 4096,
                "messages": [
                    {"role": "user", "content": prompt},
                ],
                "temperature": 0.0,
            },
        )
        return AnthropicClient.ApiResponse(
            content=response["content"][0]["text"],
            model=response.get("model", self.model),
        )


class AWSBedrockClient(Client):
    def __init__(self):
        super().__init__()

        # Check Bedrock availability with detailed error reporting
        is_available, error_msg = _check_bedrock_availability()
        if not is_available:
            if "boto3 not installed" in str(error_msg):
                raise ImportError(
                    'AI summary through AWS bedrock requires "boto3" to be installed. Install it with `pip install boto3`'
                )
            else:
                raise RuntimeError(f"AWS Bedrock is not available: {error_msg}")

        self.model = config.ai_model
        self.name = "aws_bedrock"
        self.title = "AWS Bedrock"

        import boto3  # type: ignore

        self.client = boto3.client(service_name="bedrock-runtime")

    def max_tokens(self) -> int:
        return config.ai_custom_context_window or 200000

    class ApiResponse(NamedTuple):
        content: str
        model: str

    def _query(self, prompt: str) -> ApiResponse:
        # TODO consider error-handling/backoff
        body = json.dumps(
            {
                # this is the only allowable value as of 2025/03/04
                # if they ever add more, we can make it configurable
                # or add smart logic to figure it out.
                "anthropic_version": "bedrock-2023-05-31",
                "messages": [{"role": "user", "content": [{"type": "text", "text": prompt}]}],
                "max_tokens": 4096,
            }
        )

        response = self.client.invoke_model(
            body=body, modelId=self.model, accept="application/json", contentType="application/json"
        )

        response_body = json.loads(response["body"].read())
        content = response_body["content"][0]["text"]  # Extract the assistant's response
        return AWSBedrockClient.ApiResponse(content=content, model=self.model)


class SeqeraClient(Client):
    def __init__(self, api_key: str):
        super().__init__(api_key)
        self.name = "seqera"
        self.title = "Seqera AI"
        creation_date = report.creation_date.strftime("%d %b %Y, %H:%M %Z")
        self.chat_title = f"{(config.title + ': ' if config.title else '')}MultiQC report, created on {creation_date}"
        self.tags = ["multiqc", f"multiqc_version:{config.version}"]
        self.model = config.ai_model or "claude-sonnet-4-0"

    def max_tokens(self) -> int:
        return 200000

    def wrap_details(self, prompt) -> str:
        return f"{self.chat_title}\n\n:::details\n\n{prompt}\n\n:::\n\n"

    class ApiResponse(NamedTuple):
        content: Union[str, Dict[str, str]]
        model: str
        thread_id: Optional[str] = None

    def _send_request(
        self, prompt: str, report_content: str, extra_options: Optional[Dict[str, Any]] = None
    ) -> ApiResponse:
        response = self._request_with_error_handling_and_retries(
            f"{config.seqera_api_url}/internal-ai/query",
            headers={"Authorization": f"Bearer {self.api_key}"},
            body={
                "message": self.wrap_details(prompt + "\n\n" + report_content),
                "tags": self.tags,
                "title": self.chat_title,
                **(extra_options or {}),
            },
        )
        return SeqeraClient.ApiResponse(
            content=response["generation"],
            model=response.get("model", self.model),
            thread_id=response.get("thread_id"),
        )

    def interpret_report_short(self, report_content: str) -> InterpretationResponse:
        response = self._send_request(self.prompt_short, report_content, extra_options=None)

        return InterpretationResponse(
            interpretation=InterpretationOutput(summary=str(response.content)),
            model=response.model,
            thread_id=response.thread_id,
        )

    def interpret_report_full(self, report_content: str) -> InterpretationResponse:
        response = self._send_request(
            self.prompt_full,
            report_content,
            extra_options={
                "response_schema": {
                    "name": "Interpretation",
                    "description": "Interpretation of a MultiQC report",
                    "input_schema": {
                        "type": "object",
                        "required": ["summary"],
                        "properties": {
                            key: {
                                "type": "string",
                                "description": value.description,
                                **({"default": value.default} if value.default is None else {}),
                            }
                            for key, value in InterpretationOutput.model_fields.items()
                        },
                    },
                },
            },
        )

        try:
            output = InterpretationOutput.model_validate(response.content)
            return InterpretationResponse(
                interpretation=output,
                model=response.model,
                thread_id=response.thread_id,
            )
        except Exception as e:
            logger.error(f"Failed to parse Seqera response as JSON: {e}")
            raise


def check_bedrock_availability() -> Tuple[bool, Optional[str]]:
    """
    Public function to check if AWS Bedrock is available and usable.
    Can be used for diagnostics.

    Returns:
        Tuple of (is_available, error_message)
    """
    return _check_bedrock_availability()


def _check_bedrock_availability() -> Tuple[bool, Optional[str]]:
    """
    Check if AWS Bedrock is available and usable.

    Returns:
        Tuple of (is_available, error_message)
    """
    try:
        import boto3  # type: ignore
    except ImportError:
        return False, "boto3 not installed"

    try:
        # Create bedrock runtime client
        client = boto3.client("bedrock-runtime")

        # Get the current region
        session = boto3.Session()
        region: Optional[str] = session.region_name

        # If no region is set, try to get it from the client
        if not region:
            try:
                region = client.meta.region_name
            except Exception:
                pass

        # If still no region, check environment variable
        if not region:
            region = os.environ.get("AWS_DEFAULT_REGION") or os.environ.get("AWS_REGION")

        # Default to us-east-1 if still no region found
        if not region:
            region = "us-east-1"

        # Check if Bedrock is available in this region
        # Bedrock is available in specific regions only (as of 2024)
        bedrock_regions = {
            "us-east-1",
            "us-west-2",
            "us-gov-west-1",  # US regions
            "ap-southeast-1",
            "ap-southeast-2",
            "ap-northeast-1",
            "ap-south-1",  # Asia Pacific
            "eu-central-1",
            "eu-west-1",
            "eu-west-3",  # Europe
            "ca-central-1",  # Canada
            "sa-east-1",  # South America
        }

        if region not in bedrock_regions:
            return False, f"Bedrock not available in region {region}"

        # Try to list available foundation models to verify permissions
        # This is a read-only operation that should work if user has basic Bedrock access
        bedrock_client = boto3.client("bedrock", region_name=region)
        try:
            response = bedrock_client.list_foundation_models()

            # Get list of available models
            available_models = [model.get("modelId", "") for model in response.get("modelSummaries", [])]

            # If a specific model is configured, check if it's available
            if config.ai_model:
                if config.ai_model not in available_models:
                    return (
                        False,
                        f"Configured model '{config.ai_model}' not available in Bedrock. Available models: {', '.join(available_models)}",
                    )
            else:
                # Check if any Claude models are available (since that's what we typically use)
                claude_models = [model_id for model_id in available_models if "claude" in model_id.lower()]

                if not claude_models:
                    return (
                        False,
                        f"No Claude models available in Bedrock. Available models: {', '.join(available_models)}",
                    )

            return True, None

        except Exception as e:
            if "AccessDenied" in str(e):
                return False, f"Access denied to Bedrock: {e}"
            elif "UnauthorizedOperation" in str(e):
                return False, f"Insufficient permissions for Bedrock: {e}"
            else:
                return False, f"Error checking Bedrock permissions: {e}"

    except Exception as e:
        if "NoCredentialsError" in str(e):
            return False, "AWS credentials not configured"
        elif "InvalidAccessKeyId" in str(e):
            return False, "Invalid AWS credentials"
        else:
            return False, f"Error creating Bedrock client: {e}"


def _auto_detect_provider() -> Optional[str]:
    """
    Auto-detect AI provider based on available environment variables.

    Returns the provider name in order of preference:
    1. "seqera" if SEQERA_ACCESS_TOKEN or TOWER_ACCESS_TOKEN is set
    2. "anthropic" if ANTHROPIC_API_KEY is set
    3. "openai" if OPENAI_API_KEY is set
    4. "aws_bedrock" if AWS credentials are available
    5. None if no provider can be detected
    """
    # Check for Seqera tokens
    if os.environ.get("SEQERA_ACCESS_TOKEN") or os.environ.get("TOWER_ACCESS_TOKEN"):
        return "seqera"

    # Check for Anthropic API key
    if os.environ.get("ANTHROPIC_API_KEY"):
        return "anthropic"

    # Check for OpenAI API key
    if os.environ.get("OPENAI_API_KEY"):
        return "openai"

    # Check for AWS Bedrock availability with improved detection
    is_available, error_msg = _check_bedrock_availability()
    if is_available:
        return "aws_bedrock"
    elif error_msg and config.verbose:
        logger.debug(f"AWS Bedrock not available: {error_msg}")

    return None


def get_llm_client() -> Optional[Client]:
    if not config.ai_summary:
        return None

    # Auto-detect provider if not explicitly set
    provider = config.ai_provider
    if provider is None:
        provider = _auto_detect_provider()
        if provider is None:
            raise RuntimeError(
                "config.ai_summary is set to true, but no AI provider could be detected. "
                "Please set config.ai_provider explicitly or ensure one of the following environment variables is set: "
                "SEQERA_ACCESS_TOKEN, TOWER_ACCESS_TOKEN, ANTHROPIC_API_KEY, OPENAI_API_KEY, or configure AWS credentials"
            )
        logger.debug(f"Auto-detected AI provider: {provider}")

    if provider == "seqera":
        if api_key := os.environ.get("SEQERA_ACCESS_TOKEN"):
            logger.debug("Using Seqera access token from $SEQERA_ACCESS_TOKEN environment variable")
        elif api_key := os.environ.get("TOWER_ACCESS_TOKEN"):
            logger.debug("Using Seqera access token from TOWER_ACCESS_TOKEN environment variable")
        else:
            raise RuntimeError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'seqera', "
                "but Seqera access token is not set. "
                "Please set the SEQERA_ACCESS_TOKEN / TOWER_ACCESS_TOKEN environment variable "
                "or change config.ai_provider"
            )
        return SeqeraClient(api_key)

    elif provider == "anthropic":
        api_key = os.environ.get("ANTHROPIC_API_KEY")
        if not api_key:
            raise RuntimeError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'anthropic', but Anthropic API "
                "key not set. Please set the ANTHROPIC_API_KEY environment variable, or change config.ai_provider"
            )
        logger.debug("Using Anthropic API key from ANTHROPIC_API_KEY environment variable")
        try:
            return AnthropicClient(api_key)
        except ModuleNotFoundError:
            raise ModuleNotFoundError(
                'AI summary requested through `config.ai_summary`, but required dependencies are not installed. Install them with `pip install "multiqc[anthropic]"`'
            )

    elif provider == "openai":
        api_key = os.environ.get("OPENAI_API_KEY")
        if not api_key:
            raise RuntimeError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'openai', but OpenAI API "
                "key not set. Please set the OPENAI_API_KEY environment variable, or change config.ai_provider"
            )
        logger.debug("Using OpenAI API key from OPENAI_API_KEY environment variable")
        try:
            return OpenAiClient(api_key)
        except ModuleNotFoundError:
            raise ModuleNotFoundError(
                'AI summary requested through `config.ai_summary`, but required dependencies are not installed. Install them with `pip install "multiqc[openai]"`'
            )

    elif provider == "aws_bedrock":
        try:
            return AWSBedrockClient()
        except ModuleNotFoundError:
            raise ModuleNotFoundError(
                'AI summary requested through `config.ai_summary`, but required dependencies are not installed. Install them with `pip install "multiqc[aws_bedrock]"`'
            )

    elif provider == "custom":
        api_key = os.environ.get("OPENAI_API_KEY")
        if not api_key:
            raise RuntimeError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'custom', but OpenAI API "
                "key not set. Please set the OPENAI_API_KEY environment variable, or change config.ai_provider"
            )
        if not config.ai_model:
            raise ValueError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'custom', but no config.ai_model is provided. Please set config.ai_model"
            )
        if not config.ai_custom_endpoint:
            raise ValueError(
                "config.ai_summary is set to true, and config.ai_provider is set to 'custom', but no config.ai_custom_endpoint is provided. Please set config.ai_custom_endpoint"
            )
        logger.debug(
            f"Using API key from the OPENAI_API_KEY environment variable to use with a custom endpoint {config.ai_custom_endpoint}"
        )
        return OpenAiClient(api_key=api_key, endpoint=config.ai_custom_endpoint)
    else:
        avail_providers = config_schema.AiProviderLiteral.__args__  # type: ignore
        raise RuntimeError(
            f'Unknown AI provider "{provider}". Please set config.ai_provider to one of the following: [{", ".join(avail_providers)}]'
        )


def _strip_html(text: str) -> str:
    return text.replace("<p>", "").replace("</p>", "")


class AiToolMetadata(BaseModel):
    name: str
    info: Optional[str] = None
    href: Optional[str] = None
    comment: Optional[str] = None


class AiSectionMetadata(BaseModel):
    name: str
    module_anchor: Anchor
    description: Optional[str] = None
    comment: Optional[str] = None
    helptext: Optional[str] = None
    plot_anchor: Optional[Anchor] = None
    content: Optional[str] = None
    content_before_plot: Optional[str] = None


class AiReportMetadata(BaseModel):
    tools: Dict[Anchor, AiToolMetadata]
    sections: Dict[Anchor, AiSectionMetadata]


def ai_section_metadata() -> AiReportMetadata:
    return AiReportMetadata(
        tools={
            mod.anchor: AiToolMetadata(
                name=mod.name,
                info=mod.info,
                href=", ".join(mod.href),
                comment=mod.comment,
            )
            for mod in report.modules
            if not mod.hidden and mod.name != "Software Versions"
        },
        sections={
            section.anchor: AiSectionMetadata(
                name=section.name,
                description=section.description,
                comment=section.comment,
                helptext=section.helptext,
                plot_anchor=section.plot_anchor,
                module_anchor=section.module_anchor,
                content=section.content,
                content_before_plot=section.content_before_plot,
            )
            for mod in report.modules
            for section in mod.sections
            if not mod.hidden and mod.name != "Software Versions"
        },
    )


def create_pseudonym_map(sample_names: List[SampleName]) -> Dict[str, str]:
    """
    Find all sample names in the report and replace them with anonymised names
    """
    # Create anonymised names map and reverse map
    ai_pseudonym_map = {}
    for i, name in enumerate(sample_names):
        ai_pseudonym_map[str(name)] = f"SAMPLE_{i + 1}"
    return ai_pseudonym_map


def deanonymize_sample_names(text: str) -> str:
    """
    Convert pseudonyms back to original sample names in the text.
    Only applies when config.ai_anonymize_samples is True.
    """
    if not config.ai_anonymize_samples or not report.ai_pseudonym_map:
        return text

    # Create reverse mapping from pseudonym to original name
    reverse_map = {v: k for k, v in report.ai_pseudonym_map.items()}

    # Replace pseudonyms with original names, being careful to only replace whole words
    # and preserve the directive syntax
    for pseudonym, original in reverse_map.items():
        # Look for pseudonym in :sample[SAMPLE_1]{.text-*} directives
        text = re.sub(f":sample\\[{re.escape(pseudonym)}\\](\\{{[^}}]+\\}})", f":sample[{original}]\\1", text)

        # Look for standalone pseudonyms (not in directives)
        text = re.sub(f"\\b{re.escape(pseudonym)}\\b", str(original), text)

    return text


def build_prompt(metadata: AiReportMetadata, system_prompt, client: Optional[Client] = None) -> Tuple[str, bool]:
    # Account for system message, plus leave 10% buffer
    max_tokens = client.max_tokens() if client is not None else math.inf

    user_prompt: str = ""
    current_n_tokens = client.n_tokens(system_prompt) if client is not None else 0

    # Details about modules used in the report - include first in the prompt
    tools_context: str = ""
    tools_context += "Tools used in the report:\n\n"
    for i, tool in enumerate(metadata.tools.values()):
        tools_context += f"{i + 1}. {tool.name}\n"
        if tool.info:
            tools_context += f"Description: {tool.info}\n"
        if tool.href:
            tools_context += f"Links: {tool.href}\n"
        if tool.comment:
            tools_context += f"Comment: {tool.comment}\n"
        tools_context += "\n\n"
        tools_context += "\n----------------------\n\n"
    user_prompt += tools_context

    # General stats - also always include, otherwise we don't have anything to summarize
    if general_stats_plot := report.plot_by_id.get(Anchor("general_stats_table")):
        if isinstance(general_stats_plot, Plot):
            genstats_context = f"""
    MultiQC General Statistics (overview of key QC metrics for each sample, across all tools)
    {general_stats_plot.format_for_ai_prompt()}
    """
            if client is not None:
                genstats_n_tokens = client.n_tokens(genstats_context) if client is not None else 0
                if current_n_tokens + genstats_n_tokens > max_tokens:
                    # If it's too long already, try without hidden columns
                    genstats_context = f"""
        MultiQC General Statistics (overview of key QC metrics for each sample, across all tools)
        {general_stats_plot.format_for_ai_prompt(keep_hidden=False)}
        """
                    genstats_n_tokens = client.n_tokens(genstats_context)
                    if current_n_tokens + genstats_n_tokens > max_tokens:
                        logger.debug(
                            f"General stats (almost) exceeds the {client.title}'s context window ({current_n_tokens} + "
                            f"{genstats_n_tokens} tokens, max: {client.max_tokens()} tokens). "
                            "AI summary will not be generated. Try hiding some columns in the general stats table "
                            "(see https://docs.seqera.io/multiqc/reports/customisation#hiding-columns) to reduce the context. "
                            "You can also open the HTML report in the browser, hide columns or samples dynamically, and request "
                            "the AI summary dynamically, or copy the prompt into clipboard and use it with extrenal services."
                        )
                        return user_prompt + genstats_context, True
                user_prompt += genstats_context
                current_n_tokens += genstats_n_tokens

    user_prompt = re.sub(r"\n\n\n", "\n\n", user_prompt)  # strip triple newlines

    # Build sections context and use it unless it's too long for the LLM context
    sec_context: str = ""
    for section in metadata.sections.values():
        tool = metadata.tools[section.module_anchor]
        sec_context += "\n----------------------\n\n"
        sec_context += f"Tool: {tool.name}\n"
        sec_context += f"Section: {section.name}\n"
        if section.description:
            sec_context += f"Section description: {_strip_html(section.description)}\n"
        if section.comment:
            sec_context += f"Section comment: {_strip_html(section.comment)}\n"
        if section.helptext:
            sec_context += f"Section help text: {_strip_html(section.helptext)}\n"

        if section.content_before_plot:
            sec_context += report.anonymize_sample_name(section.content_before_plot) + "\n\n"
        if section.content:
            sec_context += report.anonymize_sample_name(section.content) + "\n\n"

        if section.plot_anchor and section.plot_anchor in report.plot_by_id:
            plot = report.plot_by_id[section.plot_anchor]
            if isinstance(plot, Plot):
                if plot_content := plot.format_for_ai_prompt(keep_hidden=True):
                    if plot.pconfig.title:
                        sec_context += f"Title: {plot.pconfig.title}\n"
                    sec_context += "\n" + plot_content

        # Check if adding this section would exceed the limit
        # Using rough estimate of 4 chars per token
        if client is not None:
            sec_n_tokens = client.n_tokens(sec_context)
            if current_n_tokens + sec_n_tokens > max_tokens:
                logger.debug(
                    f"Including only General Statistics table to fit within {client.title}'s context window ({client.max_tokens()} tokens). "
                    f"Tokens estimate: {current_n_tokens}, with sections: {current_n_tokens + sec_n_tokens}"
                )
                return user_prompt, False

    user_prompt += sec_context
    user_prompt = re.sub(r"\n\n\n", "\n\n", user_prompt)  # strip triple newlines
    return user_prompt, False


def _save_prompt_to_file(system_prompt: str, prompt: str):
    """Save content to file for debugging"""
    path = report.data_tmp_dir() / "llms-full.txt"
    path.write_text(f"{system_prompt}\n\n----------------------\n\n{prompt}", encoding="utf-8")
    logger.debug(f"Saved AI prompt to {path.parent.name}/{path.name}")


def add_ai_summary_to_report():
    metadata: AiReportMetadata = ai_section_metadata()
    # Set data for JS runtime
    report.ai_report_metadata_base64 = base64.b64encode(metadata.model_dump_json().encode()).decode()
    report.ai_extra_query_options_base64 = base64.b64encode(
        json.dumps(config.ai_extra_query_options or {}).encode()
    ).decode()
    # Create and save the map for format_dataset_for_ai_prompt or JS runtime
    report.ai_pseudonym_map = create_pseudonym_map(report.sample_names)
    # Save for the JS runtime. We want to do it regardless of config.ai_anonymize_samples,
    # because JS runtime might need it even if Python runtime doesn't
    report.ai_pseudonym_map_base64 = base64.b64encode(json.dumps(report.ai_pseudonym_map).encode()).decode()

    system_prompt_short = config.ai_prompt_short or PROMPT_SHORT
    system_prompt_full = config.ai_prompt_full or PROMPT_FULL
    system_prompt = system_prompt_full if config.ai_summary_full else system_prompt_short

    # Now that everything for JS runtime is set, we only continue if static summaries are requested
    if not config.ai_summary:
        # Not generating report in Python, leaving for JS runtime, but saving full and complete prompt to llms-full.txt file
        prompt, _ = build_prompt(metadata, system_prompt)
        _save_prompt_to_file(system_prompt, prompt)
        return

    # get_llm_client() will raise an exception if configuration is invalid when ai_summary=True
    client = get_llm_client()
    assert client is not None, "get_llm_client() should not return None when config.ai_summary is True"

    report.ai_provider_id = client.name
    report.ai_provider_title = client.title
    report.ai_model = client.model

    prompt, exceeded_context_window = build_prompt(metadata, system_prompt, client)
    _save_prompt_to_file(system_prompt, prompt)
    if exceeded_context_window:
        return

    response: InterpretationResponse
    try:
        if config.ai_summary_full:
            if config.development and os.environ.get("MQC_STUB_AI_RESPONSE"):
                response = InterpretationResponse(
                    interpretation=InterpretationOutput(
                        summary=_EXAMPLE_SUMMARY_FOR_FULL,
                        detailed_analysis=_EXAMPLE_DETAILED_SUMMARY,
                    ),
                    model="test-model",
                    thread_id="68bcead8-1bea-4b75-84d1-fc2ae6afed51" if client.name == "seqera" else None,
                )
            else:
                response = client.interpret_report_full(prompt)
        else:
            if config.development and os.environ.get("MQC_STUB_AI_RESPONSE"):
                response = InterpretationResponse(
                    interpretation=InterpretationOutput(
                        summary="- All samples show :span[good quality metrics]{.text-green} with consistent CpG methylation (:span[75.7-77.0%]{.text-green}), alignment rates (:span[76-86%]{.text-green}), and balanced strand distribution (:span[~50/50]{.text-green})\n- :sample[2wk]{.text-yellow} samples show slightly higher duplication (:span[11-15%]{.text-yellow}) and trimming rates (:span[13-23%]{.text-yellow}) compared to :sample[1wk]{.text-green} samples (:span[6-9%]{.text-green} duplication, :span[2-3%]{.text-green} trimming)",
                    ),
                    model="test-model",
                    thread_id="68bcead8-1bea-4b75-84d1-fc2ae6afed51" if client.name == "seqera" else None,
                )
            else:
                response = client.interpret_report_short(prompt)
    except Exception as e:
        logger.error(f"Failed to interpret report with {client.title}: {e}")
        if config.strict:
            raise
        return None

    if not response.interpretation:
        return None

    if response.model:
        report.ai_model_resolved = response.model

    if response.thread_id:
        report.ai_thread_id = response.thread_id

    interpretation: InterpretationOutput = response.interpretation
    report.ai_global_summary = interpretation.markdown_to_html(interpretation.summary)

    if config.ai_summary_full and interpretation.detailed_analysis:
        report.ai_global_detailed_analysis = interpretation.markdown_to_html(interpretation.detailed_analysis)

    logger.info(f"Summarised report with {report.ai_provider_title}")
