##########################################################################################
# oops/hosts/hst/acs/__init__.py
##########################################################################################

import astropy.io.fits as pyfits

import oops
from .. import HST

from filecache import FCPath

##########################################################################################
# Standard class method
##########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given data file
    generated by HST/ACS.
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the ACS class
    this = ACS()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError(f'not an HST file: {filespec}')

    # Confirm that the instrument is ACS
    if this.instrument_name(hdulist) != 'ACS':
        raise IOError(f'not an HST/ACS file: {filespec}')

    return ACS.from_hdulist(hdulist, **parameters)

##########################################################################################
# ACS class
##########################################################################################

class ACS(HST):
    """This class defines functions and properties unique to the ACS instrument.
    Everything else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    # The HRC and WFC detectors have two filter wheels. Names are identified by FITS
    # parameters FILTER1 and FILTER2 in the first header. SBC overrides this method
    # because only FILTER1 is used.
    def filter_name(self, hdulist):
        """The name of the filter for this particular ACS detector. Overlapped filters are
        separated by a plus sign.
        """

        name1 = hdulist[0].header['FILTER1']
        name2 = hdulist[0].header['FILTER2']

        if name1[0:5] == 'CLEAR':
            if name2[0:5] == 'CLEAR':
                return 'CLEAR'
            else:
                return name2
        else:
            if name2[0:5] == 'CLEAR':
                return name1
            else:
                return name1 + '+' + name2

    def dn_per_sec_factor(self, hdulist):
        """The factor that converts a pixel value to DN per second.

        Input:
            hdulist     the object returned by pyfits.open()

        Return          the factor to multiply a pixel value by to get DN/sec
        """

        return 1. / hdulist[0].header['EXPTIME']

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general, static method to return an Observation object based on an HST data
        file generated by HST/ACS.
        """

        # Make an instance of the ACS class
        this = ACS()

        # Figure out the detector
        detector = this.detector_name(hdulist)

        if detector == 'HRC':
            from .hrc import HRC
            obs = HRC.from_hdulist(hdulist, **parameters)

        elif detector == 'WFC':
            from .wfc import WFC
            obs = WFC.from_hdulist(hdulist, **parameters)

        elif detector == 'SBC':
            from .sbc import SBC
            obs = SBC.from_hdulist(hdulist, **parameters)

        else:
            raise IOError('unsupported detector in HST/ACS file ' +
                          this.filespec(hdulist) + ': ' + detector)

        return obs

##########################################################################################
