################################################################################
# oops/hosts/hst/acs/sbc.py
################################################################################

import astropy.io.fits as pyfits

from . import ACS

from filecache import FCPath

################################################################################
# Standard class methods
################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given
    data file generated by HST/ACS/SBC.
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the SBC class
    this = SBC()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError(f'not an HST file: {filespec}')

    # Confirm that the instrument is ACS
    if this.instrument_name(hdulist) != 'ACS':
        raise IOError(f'not an HST/ACS file: {filespec}')

    # Confirm that the detector is SBC
    if this.detector_name(hdulist) != 'SBC':
        raise IOError(f'not an HST/ACS/SBC file: {filespec}')

    return SBC.from_opened_fitsfile(hdulist, **parameters)

IDC_DICT = None

GENERAL_SYN_FILES = ['OTA/hst_ota_???_syn.fits',
                     'ACS/acs_sbc_mama_???_syn.fits']

FILTER_SYN_FILE = ['ACS/acs_', '_???_syn.fits']

#===============================================================================
#===============================================================================
class SBC(ACS):
    """This class defines functions and properties unique to the NIC1 detector.
    Everything else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    def define_fov(self, hdulist, **parameters):
        """An FOV object defining the field of view of the given image file."""

        global IDC_DICT

        # Load the dictionary of IDC parameters if necessary
        if IDC_DICT is None:
            IDC_DICT = self.load_idc_dict(hdulist, ('FILTER1',))

        # Define the key into the dictionary
        idc_key = (hdulist[0].header['FILTER1'],)

        # Use the default function defined at the HST level for completing the
        # definition of the FOV
        return self.construct_fov(IDC_DICT[idc_key], hdulist)

    #===========================================================================
    def filter_name(self, hdulist, layer=None):
        """The name of the filter for this particular ACS detector."""

        return hdulist[0].header['FILTER1']

    #===========================================================================
    def select_syn_files(self, hdulist, **parameters):
        """The list of SYN files containing profiles that are to be multiplied
        together to obtain the throughput of the given instrument, detector, and
        filter combination.
        """

        # Copy all the standard file names
        syn_filenames = []
        for filename in GENERAL_SYN_FILES:
            syn_filenames.append(filename)

        # Add the filter file name
        syn_filenames.append(FILTER_SYN_FILE[0] +
                             hdulist[0].header['FILTER1'].lower() +
                             FILTER_SYN_FILE[1])

        return syn_filenames

    #===========================================================================
    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general class method to return an Observation object based on an
        HST data file generated by HST/ACS/SBC.
        """

        return SBC().construct_snapshot(hdulist, **parameters)

################################################################################
