##########################################################################################
# oops/hosts/hst/acs/wfc.py
##########################################################################################

import astropy.io.fits as pyfits

from . import ACS

from filecache import FCPath

##########################################################################################
# Classless method
##########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given data file
    generated by HST/ACS/WFC.

    For WFC,
        ccd         1 for WFC1; 2 for WFC3
        layer       1 for the first image in FITS file, 2 for the second.

    Note: Either layer or ccd must be specified. If both are specified, they
    must be compatible
    """

    # Open the file if necessary
    if isinstance(filespec, str):
        filespec = FCPath(filespec)

        # Open the file
        local_path = filespec.retrieve()
        hdulist = pyfits.open(local_path)
    else:
        hdulist = filespec

    # Make an instance of the WFC class
    this = WFC()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError(f'not an HST file: {filespec}')

    # Confirm that the instrument is ACS
    if this.instrument_name(hdulist) != 'ACS':
        raise IOError(f'not an HST/ACS file: {filespec}')

    # Confirm that the detector is WFC
    if this.detector_name(hdulist) != 'WFC':
        raise IOError(f'not an HST/ACS/WFC file: {filespec}')

    return WFC.from_opened_fitsfile(hdulist, **parameters)

##########################################################################################
# Class WFC
##########################################################################################

IDC_DICT = None

GENERAL_SYN_FILES = ['OTA/hst_ota_???_syn.fits', 'ACS/acs_wfc_im123_???_syn.fits']

CCD_SYN_FILE_PARTS    = ['ACS/acs_wfc_ccd', '_???_syn.fits']
FILTER_SYN_FILE_PARTS = ['ACS/acs_', '_???_syn.fits']


class WFC(ACS):
    """This class defines unique detector properties of the ACS/WFC. Other properties are
    inherited from higher levels in the class hierarcy. All functions are static so this
    class requires no instances.
    """

    def define_fov(self, hdulist, **parameters):
        """An FOV object defining the field of view of the given image file."""

        global IDC_DICT

        # Load the dictionary of IDC parameters if necessary
        if IDC_DICT is None:
            IDC_DICT = self.load_idc_dict(hdulist, ('DETCHIP', 'FILTER1', 'FILTER2'))

        # Determine the layer of the FITS file to read
        layer = parameters['layer']

        # Define the key into the dictionary
        idc_key = (hdulist[layer].header['CCDCHIP'], hdulist[0].header['FILTER1'],
                                                     hdulist[0].header['FILTER2'])

        # Use the default function defined at the HST level for completing the
        # definition of the FOV
        return self.construct_fov(IDC_DICT[idc_key], hdulist)

    def select_syn_files(self, hdulist, **parameters):
        """The list of SYN files containing profiles that are to be multiplied
        together to obtain the throughput of the given instrument, detector and
        filter combination.
        """

        global GENERAL_SYN_FILES, CCD_SYN_FILE_PARTS, FILTER_SYN_FILE_PARTS

        # Copy all the standard file names
        syn_filenames = []
        for filename in GENERAL_SYN_FILES:
            syn_filenames.append(filename)

        # Add the filter file names
        for filter_name in (hdulist[0].header['FILTER1'], hdulist[0].header['FILTER2']):

            if filter_name[0:3] == 'POL':
                if filter_name[-2:] == 'UV':
                    filter_name = 'POL_UV'
                else:
                    filter_name = 'POL_V'

            if filter_name[0:5] != 'CLEAR':
                syn_filenames.append(FILTER_SYN_FILE_PARTS[0] + filter_name.lower() +
                                     FILTER_SYN_FILE_PARTS[1])

        # Determine the layer of the FITS file to read
        layer = parameters['layer']

        # Add the CCD file name
        syn_filenames.append(CCD_SYN_FILE_PARTS[0] +
                             str(hdulist[layer].header['CCDCHIP']) +
                             CCD_SYN_FILE_PARTS[1])

        return syn_filenames

    def register_frame(self, hdulist, fov, **parameters):
        """Override the default to define a suffix for the name of the frame.

        Needed so that names for WFC1 and WFC2 are unique.
        """

        ccd = parameters['ccd']

        return super(WFC,self).register_frame(hdulist, fov, suffix='_WFC' + str(ccd),
                                              **parameters)

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general class method to return an Observation object based on an HST data
        file generated by HST/ACS/WFC.
        """

        # Make an instance of the WFC class
        this = WFC()

        # Figure out the layer and CCD; make sure they are compatible
        if 'layer' in parameters:
            given_layer = parameters['layer']
            derived_ccd = hdulist[given_layer].header['CCDCHIP']
        else:
            given_layer = None
            derived_ccd = None

        if 'ccd' in parameters:
            given_ccd = parameters['ccd']

            try:
                for layer in range(1,6):        # IndexError on 5
                    try:
                        test_ccd = hdulist[layer].header['CCDCHIP']
                        if test_ccd == given_ccd:
                            derived_layer = layer
                            break
                    except KeyError:
                        pass

            except IndexError:
                raise ValueError(f'CCD {given_ccd} not found: ' + this.filespec(hdulist))

        else:
            given_ccd = None
            derived_layer = None

        ccd = given_ccd or derived_ccd
        layer = given_layer or derived_layer

        if ccd is None or layer is None:
            raise ValueError('layer and ccd parameters are undefined: ' +
                             this.filespec(hdulist))

        if derived_ccd not in (None, ccd) or \
           derived_layer not in (None, layer):
            raise ValueError('layer and ccd parameters are incompatible: ' +
                             this.filespec(hdulist))

        parameters = parameters.copy()
        parameters['ccd'] = ccd
        parameters['layer'] = layer

        obs = this.construct_snapshot(hdulist, **parameters)

        # Insert subfields common to all images
        obs.insert_subfield('layer', layer)
        obs.insert_subfield('chip', ccd)

        return obs

##########################################################################################
