#########################################################################################
# oops/hosts/hst/nicmos/nic1.py: HST/NICMOS subclass NIC1
#########################################################################################

try:
    import astropy.io.fits as pyfits
except ImportError:
    import pyfits
from . import NICMOS

from filecache import FCPath

#########################################################################################
# Standard class method
#########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given
    data file generated by HST/NICMOS/NIC1.
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the NIC1 class
    this = NIC1()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError(f'not an HST file: {filespec}')

    # Confirm that the instrument is NICMOS
    if this.instrument_name(hdulist) != 'NICMOS':
        raise IOError(f'not an HST/NICMOS file: {filespec}')

    # Confirm that the detector is NIC1
    if this.detector_name(hdulist) != 'IR':
        raise IOError(f'not an HST/NICMOS/NIC1 file: {filespec}')

    return NIC1.from_opened_fitsfile(hdulist, **parameters)

#########################################################################################
# NIC1 class
#########################################################################################

class NIC1(NICMOS):
    """This class defines functions and properties unique to the NIC1 detector. Everything
    else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    # Used by select_syn_files, defined in NICMOS.py
    DETECTOR_SYN_FILES = ['NICMOS/nic1_bend_???_syn.fits',
                          'NICMOS/nic1_cmask_???_syn.fits',
                          'NICMOS/nic1_dewar_???_syn.fits',
                          'NICMOS/nic1_image_???_syn.fits',
                          'NICMOS/nic1_para1_???_syn.fits',
                          'NICMOS/nic1_para2_???_syn.fits']

    FILTER_SYN_FILE_PARTS = ['NICMOS/nic1_', '_???_syn.fits']

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general class method to return an Observation object based on an
        HST data file generated by HST/NICMOS/NIC1.
        """

        return NIC1().construct_snapshot(hdulist, **parameters)

#########################################################################################
