##########################################################################################
# oops/hosts/hst/nicmos/nic2.py
##########################################################################################

import astropy.io.fits as pyfits

from . import NICMOS

from filecache import FCPath

##########################################################################################
# Standard class method
##########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given
    data file generated by HST/NICMOS/NIC1.
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the NIC2 class
    this = NIC2()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError(f'not an HST file: {filespec}')

    # Confirm that the instrument is NICMOS
    if this.instrument_name(hdulist) != 'NICMOS':
        raise IOError(f'not an HST/NICMOS file: {filespec}')

    # Confirm that the detector is NIC2
    if this.detector_name(hdulist) != 'IR':
        raise IOError(f'not an HST/NICMOS/NIC2 file: {filespec}')

    return NIC2.from_opened_fitsfile(hdulist, **parameters)

##########################################################################################
# NIC2 class
##########################################################################################

class NIC2(NICMOS):
    """This class defines functions and properties unique to the NIC2 detector. Everything
    else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    # Used by select_syn_files, defined in NICMOS.py
    DETECTOR_SYN_FILES = ['NICMOS/nic2_bend_???_syn.fits',
                          'NICMOS/nic2_cmask_???_syn.fits',
                          'NICMOS/nic2_dewar_???_syn.fits',
                          'NICMOS/nic2_image_???_syn.fits',
                          'NICMOS/nic2_para1_???_syn.fits',
                          'NICMOS/nic2_para2_???_syn.fits']

    FILTER_SYN_FILE_PARTS = ['NICMOS/nic2_', '_???_syn.fits']

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general class method to return an Observation object based on an HST data
        file generated by HST/NICMOS/NIC1.
        """

        return NIC2().construct_snapshot(hdulist, **parameters)

##########################################################################################
