##########################################################################################
# oops/hosts/hst/wfc3/__init__.py: HST subclass WFC3
##########################################################################################

import astropy.io.fits as pyfits
import oops
from .. import HST

from filecache import FCPath

##########################################################################################
# Standard class method
##########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given
    data file generated by HST/WFC3.
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the WFC3 class
    this = WFC3()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError('not an HST file: ' + this.filespec(hdulist))

    # Confirm that the instrument is ACS
    if this.instrument_name(hdulist) != 'WFC3':
        raise IOError('not an HST/WFC3 file: ' + this.filespec(hdulist))

    return WFC3.from_opened_fitsfile(hdulist)


class WFC3(HST):
    """This class defines functions and properties unique to the WFC3 instrument.
    Everything else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    #  Both WFC3 detectors have a single filter wheel. The name is identified by
    #  FITS parameter FILTER in the first header.
    def filter_name(self, hdulist):
        """The name of the filter for this particular NICMOS detector."""

        return hdulist[0].header['FILTER']

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general class method to return an Observation object based on an HST data
        file generated by HST/WFC3.
        """

        # Make an instance of the WFC3 class
        this = WFC3()

        # Figure out the detector
        detector = this.detector_name(hdulist)

        if detector == 'UVIS':
            from .uvis import UVIS
            obs = UVIS.from_hdulist(hdulist, **parameters)

        elif detector == 'IR':
            from .ir import IR
            obs = IR.from_hdulist(hdulist, **parameters)

        else:
            raise IOError('unsupported detector in HST/WFC3 file ' +
                          this.filespec(hdulist) + ': ' + detector)

        return obs

##########################################################################################
