##########################################################################################
# oops/hosts/hst/wfpc2.py
##########################################################################################

import os
import astropy.io.fits as pyfits

import oops
from . import HST

from filecache import FCPath

##########################################################################################
# Standard class method
##########################################################################################

def from_file(filespec, **parameters):
    """A general, static method to return an Observation object based on a given data file
    generated by HST/WFPC2.

    Parameters:
        ccd (int): 1 for a the PC1 image; 2-4 for WF2-WF4.
        layer (int): 1 for the first image in FITS file, 2 for the second, etc.

    Note:
        Either layer or ccd can be specified. If both are specified, they must be
        compatible. If neither is specified, layer=1 is assumed
    """

    filespec = FCPath(filespec)

    # Open the file
    local_path = filespec.retrieve()
    hdulist = pyfits.open(local_path)

    # Make an instance of the NICMOS class
    this = WFPC2()

    # Confirm that the telescope is HST
    if this.telescope_name(hdulist) != 'HST':
        raise IOError('not an HST file: ' + this.filespec(hdulist))

    # Confirm that the instrument is WFPC2
    if this.instrument_name(hdulist) != 'WFPC2':
        raise IOError('not an HST/WFPC2 file: ' + this.filespec(hdulist))

    return WFPC2.from_opened_fitsfile(hdulist, **parameters)

##########################################################################################
# WFPC2 class
##########################################################################################

IDC_DICT = None

GENERAL_SYN_FILES = ['OTA/hst_ota_???_syn.fits', 'WFPC2/wfpc2_optics_???_syn.fits']

FILTER_SYN_FILE_PARTS = ['WFPC2/wfpc2_', '_???_syn.fits']

class WFPC2(HST):
    """This class defines functions and properties unique to the WFPC2 instrument.
    Everything else is inherited from higher levels in the class hierarchy.

    Objects of this class are empty; they only exist to support inheritance.
    """

    # WFPC2 is treated as a single detector
    def detector_name(self, hdulist, **parameters):
        """The name of the detector on the HST instrument that was used to obtain this
        file. Always blank for WFPC2.
        """

        return ''

    # WFPC2 has two filter wheels. Names are identified by FITS parameters FILTNAM1 and
    # FILTNAM2 in the first header.
    def filter_name(self, hdulist):
        """The name of the filter for this particular ACS detector. Overlapped filters are
        separated by a plus sign.
        """

        name1 = hdulist[0].header['FILTNAM1'].strip()
        name2 = hdulist[0].header['FILTNAM2'].strip()

        if name1 == '':
            if name2[0:5] == '':
                return 'CLEAR'
            else:
                return name2
        else:
            if name2 == '':
                return name1
            else:
                return name1 + '+' + name2

    def data_array(self, hdulist, **parameters):
        """Array containing the data."""

        return hdulist[parameters['layer']].data

    def error_array(self, hdulist, **parameters):
        """Array containing the uncertainty values associated with the data."""

        return None      # Not available for WFPC2

    def quality_mask(self, hdulist, **parameters):
        """Array containing the quality mask.

        Use parameter 'layer' to identify the image layer in the FITS file; the quality
        mask layer will be at the same layer in the associated file.

        parameters['mask'] =
            'omit to omit mask;
            'require' or 'required' to raise an IOError if mask is unavailabl;
            'optional' to include it if available, otherwise return None

        Default is 'optional'.
        """

        # Unlike the other HST instruments, WFPC2 masks are saved in a separate
        # file. We attempt to open the mask file under the assumption that it is
        # in the same directory as the data file.
        #
        # Quality masks for raw files ("*d0m.fits") end in "q0m.fits".
        # Quality masks for calibrated files ("*c0m.fits") end in "c1m.fits".

        QUALITY_MASK_LOOKUP = {'d0':'q0', 'D0':'Q0', 'c0':'c1', 'C0':'C1'}

        if 'mask' in parameters:
            mask_option = parameters['mask']
        else:
            mask_option = 'optional'

        if mask_option == 'omit':
            return

        if mask_option not in {'require', 'required', 'optional'}:
            raise ValueError(f'Illegal value for "mask" parameter: "{mask_option}"')

        # Get the full path to the image file
        data_filespec = self.filespec(hdulist)

        # Extract the extension
        (head,tail) = os.path.splitext(data_filespec)

        # Attempt to define the mask filespec
        data_tag = head[-3:-1]
        try:
            mask_tag = QUALITY_MASK_LOOKUP[data_tag]
        except KeyError:
            if mask_option == 'optional':
                return None
            else:
                raise IOError('Unable to identify mask file for ' + data_filespec)

        # Attempt to load and return the mask
        mask_filespec = head[:-3] + mask_tag + head[-1] + tail
        try:
            f = pyfits.open(mask_filespec)
            mask_array = f[parameters['layer']].data
            f.close()

            return mask_array

        except IOError:
            if mask_option == 'optional':
                return None
            else:
                raise IOError('WFPC2 mask file not found: ' + mask_filespec)

    def register_frame(self, hdulist, fov, **parameters):
        """The Tracker frame that rotates from J2000 coordinates into the frame
        of the HST observation.
        """

        CHIPS = ['', 'PC1', 'WF2', 'WF3', 'WF4']

        ccd = parameters['ccd']

        return super(WFPC2,self).register_frame(hdulist, fov,
                                                suffix = '_' + CHIPS[ccd],
                                                **parameters)

    #===========================================================================
    # The IDC dictionaries for WFPC2 are keyed by (FILTNAM1, FILTNAM2, DETCHIP)
    def define_fov(self, hdulist, **parameters):
        """An FOV object defining the field of view of the given image file and ccd, where
        ccd values 1-4 refer to PC1, WF2, WF3 and WF4.
        """

        global IDC_DICT

        # Load the dictionary of IDC parameters if necessary
        if IDC_DICT is None:
            IDC_DICT = self.load_idc_dict(hdulist, ('FILTER1', 'FILTER2', 'DETCHIP'))

            # The IDC_DICT parameters for WFPC2 need to be re-scaled!
            for entry in IDC_DICT.keys():
                dict = IDC_DICT[entry]
                scale = dict['SCALE']
                for key in dict.keys():
                    if key[0:2] in ('CX', 'CY') and len(key) == 4:
                        dict[key] *= scale

        # Define the key into the dictionary
        filtnam1 = hdulist[0].header['FILTNAM1'].strip()
        filtnam2 = hdulist[0].header['FILTNAM2'].strip()
        if filtnam1 == '':
            filtnam1 = 'CLEAR'
        if filtnam2 == '':
            filtnam2 = 'CLEAR'

        ccd = parameters['ccd']
        idc_key = (filtnam1, filtnam2, ccd)

        # Define the FOV
        fov = super(WFPC2,self).construct_fov(IDC_DICT[idc_key], hdulist)

        # Handle AREA mode
        if hdulist[0].header['MODE'] == 'AREA':
            fov = oops.fov.Subsampled(fov, 2)

        return fov

    def select_syn_files(self, hdulist, **parameters):
        """The list of SYN files containing profiles that are to be multiplied together to
        obtain the throughput of the given instrument, detector, and filter combination.
        """

        global GENERAL_SYN_FILES, FILTER_SYN_FILE_PARTS

        # Copy all the standard file names
        syn_filenames = []
        for filename in GENERAL_SYN_FILES:
            syn_filenames.append(filename)

        # Add the filter file names
        for filter_name in (hdulist[0].header['FILTNAM1'], hdulist[0].header['FILTNAM2']):
            filter_name = filter_name.strip()
            if filter_name != '':
                syn_filenames.append(FILTER_SYN_FILE_PARTS[0] + filter_name.lower() +
                                     FILTER_SYN_FILE_PARTS[1])

        return syn_filenames

    def dn_per_sec_factor(self, hdulist):
        """The factor that converts a pixel value to DN per second.

        Input:
            hdulist    the object returned by pyfits.open()

        Return          the factor to multiply a pixel value by to get DN/sec
        """

        return 1. / hdulist[0].header['EXPTIME']

    def get_headers(self, hdulist, **parameters):
        """A list of header dictionaries from the FITS file.

        For WFPC2, there is one layer per object, not three.
        """

        headers = [hdulist[0].header]

        if 'layer' in parameters:
            layer = parameters['layer']
        else:
            layer = 1

        headers.append(hdulist[layer].header)
        return headers

    @staticmethod
    def from_hdulist(hdulist, **parameters):
        """A general, static method to return an Observation object based on an HST data
        file generated by HST/WFPC2.
        """

        # Check parameters
        if 'ccd' not in parameters and 'layer' not in parameters:
            parameters = parameters.copy()
            parameters['layer'] = 1

        # Make an instance of the WFPC2 class
        this = WFPC2()

        # Figure out the layer and CCD; make sure they are compatible
        if 'layer' in parameters:
            given_layer = parameters['layer']
            derived_ccd = hdulist[given_layer].header['DETECTOR']
        else:
            given_layer = None
            derived_ccd = None

        if 'ccd' in parameters:
            given_ccd = parameters['ccd']

            try:
                for layer in range(1,6):        # IndexError on 5
                    test_ccd = hdulist[layer].header['DETECTOR']
                    if test_ccd == given_ccd:
                        derived_layer = layer
                        break
            except IndexError:
                raise ValueError(f'CCD {given_ccd} not found: ' + this.filespec(hdulist))
        else:
            given_ccd = None
            derived_layer = None

        ccd = given_ccd or derived_ccd
        layer = given_layer or derived_layer

        if derived_ccd not in (None, ccd) or \
           derived_layer not in (None, layer):
            raise ValueError('layer and ccd parameters are incompatible: ' +
                             this.filespec(hdulist))

        parameters = parameters.copy()
        parameters['ccd'] = ccd
        parameters['layer'] = layer

        obs = this.construct_snapshot(hdulist, **parameters)

        # Insert subfields common to all images
        obs.insert_subfield('layer', layer)
        obs.insert_subfield('chip', ccd)        # old name
        obs.insert_subfield('ccd', ccd)

        return obs

##########################################################################################
