# -*- coding: utf-8 -*-
# Author: Dylan Jones
# Date:   2025-05-01

from datetime import datetime
from typing import Any, Dict, Iterable, List, Optional, Tuple

class Error(Exception):
    """The base class of the other exceptions in this module.

    Use this to catch all errors with one single except statement
    """

    ...

class DatabaseError(Exception):
    """Exception raised for errors that are related to the master.db database."""

    ...

class AnlzError(Exception):
    """Exception raised for errors that are related to the ANLZ files."""

    ...

class XmlError(Exception):
    """Exception raised for errors that are related to the Rekordbox XML handler."""

    ...

class SettingError(Exception):
    """Exception raised for errors that are related to the Rekordbox MySetting handler."""

    ...

# ---- MasterDb ------------------------------------------------------------------------------------

def is_rekordbox_running() -> bool:
    """Check if Rekordbox is running."""
    ...

class _RustMutableMapping:
    def __len__(self) -> int:
        """Return the number of items."""
    ...

    def __iter__(self) -> Iterable[str]:
        """Return an iterator over the keys."""
        ...

    def __getitem__(self, key: str) -> Any:
        """Get an item by key."""
        ...

    def __setitem__(self, key: str, value: Any) -> None:
        """Set an item by key."""
        ...

    def keys(self) -> Iterable[str]:
        """Return an iterator over the keys."""
        ...

    def values(self) -> Iterable[Any]:
        """Return an iterator over the values."""
        ...

    def items(self) -> Iterable[Tuple[str, Any]]:
        """Return an iterator over the keys and values."""
        ...

    def get(self, key: str, default: Any = None) -> Any:
        """Get an item by key or return the default value, if the key doesn't exist."""
        ...

    def update(self, data: Dict[str, Any]) -> None:
        """Update the fields from a dictionary."""
        ...

    def to_dict(self) -> Dict[str, Any]:
        """Convert the object to a dictionary."""
        ...

class _StatsFull:
    ID: str
    UUID: str
    rb_data_status: int
    rb_local_data_status: int
    rb_local_deleted: int
    rb_local_synced: int
    created_at: datetime
    updated_at: datetime
    usn: Optional[int]
    rb_local_usn: Optional[int]

class AgentRegistry(_RustMutableMapping):
    registry_id: str
    created_at: str
    updated_at: str
    id_1: Optional[str]
    id_2: Optional[str]
    int_1: Optional[int]
    int_2: Optional[int]
    str_1: Optional[str]
    str_2: Optional[str]
    date_1: Optional[str]
    date_2: Optional[str]
    text_1: Optional[str]
    text_2: Optional[str]

class CloudAgentRegistry(_StatsFull, _RustMutableMapping):
    int_1: Optional[int]
    int_2: Optional[int]
    str_1: Optional[str]
    str_2: Optional[str]
    date_1: Optional[str]
    date_2: Optional[str]
    text_1: Optional[str]
    text_2: Optional[str]

class ContentActiveCensor(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    ActiveCensors: Optional[str]
    rb_activecensor_count: Optional[int]

class ContentCue(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    Cues: Optional[str]
    rb_cue_count: Optional[int]

class ContentFile(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    Path: Optional[str]
    Hash: Optional[str]
    Size: Optional[int]
    rb_local_path: Optional[str]
    rb_insync_hash: Optional[str]
    rb_insync_local_usn: Optional[int]
    rb_file_hash_dirty: Optional[int]
    rb_local_file_status: Optional[int]
    rb_in_progress: Optional[int]
    rb_process_type: Optional[int]
    rb_temp_path: Optional[str]
    rb_priority: Optional[int]
    rb_file_size_dirty: Optional[int]

class DjmdActiveCensor(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    InMsec: Optional[int]
    OutMsec: Optional[int]
    Info: Optional[int]
    ParameterList: Optional[str]
    ContentUUID: Optional[str]

class DjmdAlbum(_StatsFull, _RustMutableMapping):
    Name: Optional[str]
    AlbumArtistID: Optional[str]
    ImagePath: Optional[str]
    Compilation: Optional[int]
    SearchStr: Optional[str]

class DjmdArtist(_StatsFull, _RustMutableMapping):
    Name: Optional[str]
    SearchStr: Optional[str]

class DjmdCategory(_StatsFull, _RustMutableMapping):
    MenuItemID: Optional[str]
    Seq: Optional[int]
    Disable: Optional[int]
    InfoOrder: Optional[int]

class DjmdColor(_StatsFull, _RustMutableMapping):
    ColorCode: Optional[str]
    SortKey: Optional[int]
    Commnt: Optional[str]

class DjmdContent(_StatsFull, _RustMutableMapping):
    FolderPath: Optional[str]
    FileNameL: Optional[str]
    FileNameS: Optional[str]
    Title: Optional[str]
    ArtistID: Optional[str]
    AlbumID: Optional[str]
    GenreID: Optional[str]
    BPM: Optional[int]
    Length: Optional[int]
    TrackNo: Optional[int]
    BitRate: Optional[int]
    BitDepth: Optional[int]
    Commnt: Optional[str]
    FileType: Optional[int]
    Rating: Optional[int]
    ReleaseYear: Optional[int]
    RemixerID: Optional[str]
    LabelID: Optional[str]
    OrgArtistID: Optional[str]
    KeyID: Optional[str]
    StockDate: Optional[str]
    ColorID: Optional[str]
    DJPlayCount: Optional[int]
    ImagePath: Optional[str]
    MasterDBID: Optional[str]
    MasterSongID: Optional[str]
    AnalysisDataPath: Optional[str]
    SearchStr: Optional[str]
    FileSize: Optional[int]
    DiscNo: Optional[int]
    ComposerID: Optional[str]
    Subtitle: Optional[str]
    SampleRate: Optional[int]
    DisableQuantize: Optional[int]
    Analysed: Optional[int]
    ReleaseDate: Optional[str]
    DateCreated: Optional[str]
    ContentLink: Optional[int]
    Tag: Optional[str]
    ModifiedByRBM: Optional[str]
    HotCueAutoLoad: Optional[str]
    DeliveryControl: Optional[str]
    DeliveryComment: Optional[str]
    CueUpdated: Optional[str]
    AnalysisUpdated: Optional[str]
    TrackInfoUpdated: Optional[str]
    Lyricist: Optional[str]
    ISRC: Optional[str]
    SamplerTrackInfo: Optional[int]
    SamplerPlayOffset: Optional[int]
    SamplerGain: Optional[float]
    VideoAssociate: Optional[str]
    LyricStatus: Optional[int]
    ServiceID: Optional[int]
    OrgFolderPath: Optional[str]
    Reserved1: Optional[str]
    Reserved2: Optional[str]
    Reserved3: Optional[str]
    Reserved4: Optional[str]
    ExtInfo: Optional[str]
    rb_file_id: Optional[str]
    DeviceID: Optional[str]
    rb_LocalFolderPath: Optional[str]
    SrcID: Optional[str]
    SrcTitle: Optional[str]
    SrcArtistName: Optional[str]
    SrcAlbumName: Optional[str]
    SrcLength: Optional[int]

class DjmdCue(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    InMsec: Optional[int]
    InFrame: Optional[int]
    InMpegFrame: Optional[int]
    InMpegAbs: Optional[int]
    OutMsec: Optional[int]
    OutFrame: Optional[int]
    OutMpegFrame: Optional[int]
    OutMpegAbs: Optional[int]
    Kind: Optional[int]
    Color: Optional[int]
    ColorTableIndex: Optional[int]
    ActiveLoop: Optional[int]
    Comment: Optional[str]
    BeatLoopSize: Optional[int]
    CueMicrosec: Optional[int]
    InPointSeekInfo: Optional[str]
    OutPointSeekInfo: Optional[str]
    ContentUUID: Optional[str]

class DjmdDevice(_StatsFull, _RustMutableMapping):
    MasterDBID: Optional[str]
    Name: Optional[str]

class DjmdGenre(_StatsFull, _RustMutableMapping):
    Name: Optional[str]

class DjmdHistory(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]
    DateCreated: Optional[str]

class DjmdSongHistory(_StatsFull, _RustMutableMapping):
    HistoryID: Optional[str]
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdHotCueBanklist(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    ImagePath: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]

class DjmdSongHotCueBanklist(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    TrackNo: Optional[int]
    CueID: Optional[str]
    InMsec: Optional[int]
    InFrame: Optional[int]
    InMpegFrame: Optional[int]
    InMpegAbs: Optional[int]
    OutMsec: Optional[int]
    OutFrame: Optional[int]
    OutMpegFrame: Optional[int]
    OutMpegAbs: Optional[int]
    Color: Optional[int]
    ColorTableIndex: Optional[int]
    ActiveLoop: Optional[int]
    Comment: Optional[str]
    BeatLoopSize: Optional[int]
    CueMicrosec: Optional[int]
    InPointSeekInfo: Optional[str]
    OutPointSeekInfo: Optional[str]
    HotCueBanklistUUID: Optional[str]

class DjmdKey(_StatsFull, _RustMutableMapping):
    ScaleName: Optional[str]
    Seq: Optional[int]

class DjmdLabel(_StatsFull, _RustMutableMapping):
    Name: Optional[str]

class DjmdMenuItems(_StatsFull, _RustMutableMapping):
    Class: Optional[int]
    Name: Optional[str]

class DjmdMixerParam(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    GainHigh: Optional[int]
    GainLow: Optional[int]
    PeakHigh: Optional[int]
    PeakLow: Optional[int]

class DjmdMyTag(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]

class DjmdSongMyTag(_StatsFull, _RustMutableMapping):
    MyTagID: Optional[str]
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdPlaylist(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    ImagePath: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]
    SmartList: Optional[str]

class DjmdPlaylistTreeItem(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    ImagePath: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]
    SmartList: Optional[str]

    Children: List["DjmdPlaylistTreeItem"]

class DjmdSongPlaylist(_StatsFull, _RustMutableMapping):
    PlaylistID: Optional[str]
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdProperty(_RustMutableMapping):
    DBID: str
    created_at: str
    updated_at: str
    DBVersion: Optional[str]
    BaseDBDrive: Optional[str]
    CurrentDBDrive: Optional[str]
    DeviceID: Optional[str]
    Reserved1: Optional[str]
    Reserved2: Optional[str]
    Reserved3: Optional[str]
    Reserved4: Optional[str]
    Reserved5: Optional[str]

class DjmdCloudProperty(_StatsFull, _RustMutableMapping):
    Reserved1: Optional[str]
    Reserved2: Optional[str]
    Reserved3: Optional[str]
    Reserved4: Optional[str]
    Reserved5: Optional[str]

class DjmdRecommendLike(_StatsFull, _RustMutableMapping):
    ContentID1: Optional[str]
    ContentID2: Optional[str]
    LikeRate: Optional[int]
    DataCreatedH: Optional[int]
    DataCreatedL: Optional[int]

class DjmdRelatedTracks(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]
    Criteria: Optional[str]

class DjmdSongRelatedTracks(_StatsFull, _RustMutableMapping):
    RelatedTracksID: Optional[str]
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdSampler(_StatsFull, _RustMutableMapping):
    Seq: Optional[int]
    Name: Optional[str]
    Attribute: Optional[int]
    ParentID: Optional[str]

class DjmdSongSampler(_StatsFull, _RustMutableMapping):
    SamplerID: Optional[str]
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdSongTagList(_StatsFull, _RustMutableMapping):
    ContentID: Optional[str]
    TrackNo: Optional[int]

class DjmdSort(_StatsFull, _RustMutableMapping):
    MenuItemID: Optional[str]
    Seq: Optional[int]
    Disable: Optional[int]

class HotCueBanklistCue(_StatsFull, _RustMutableMapping):
    HotCueBanklistID: Optional[str]
    Cues: Optional[str]
    rb_cue_count: Optional[int]

class ImageFile(_StatsFull, _RustMutableMapping):
    TableName: Optional[str]
    TargetUUID: Optional[str]
    TargetID: Optional[str]
    Path: Optional[str]
    Hash: Optional[str]
    Size: Optional[int]
    rb_local_path: Optional[str]
    rb_insync_hash: Optional[str]
    rb_insync_local_usn: Optional[int]
    rb_file_hash_dirty: Optional[int]
    rb_local_file_status: Optional[int]
    rb_in_progress: Optional[int]
    rb_process_type: Optional[int]
    rb_temp_path: Optional[str]
    rb_priority: Optional[int]
    rb_file_size_dirty: Optional[int]

class SettingFile(_StatsFull, _RustMutableMapping):
    Path: Optional[str]
    Hash: Optional[str]
    Size: Optional[int]
    rb_local_path: Optional[str]
    rb_insync_hash: Optional[str]
    rb_insync_local_usn: Optional[int]
    rb_file_hash_dirty: Optional[int]
    rb_file_size_dirty: Optional[int]

class UuidIDMap(_StatsFull, _RustMutableMapping):
    TableName: Optional[str]
    TargetUUID: Optional[str]
    CurrentID: Optional[str]

class PyMasterDb:
    def __init__(self, path: str) -> None: ...
    @classmethod
    def open(cls) -> "PyMasterDb": ...
    def set_unsafe_writes(self, bool: bool) -> None: ...
    def get_agent_registry(self) -> List[AgentRegistry]: ...
    def get_agent_registry_by_id(self, registry_id: str) -> Optional[AgentRegistry]: ...
    def get_local_usn(self) -> int: ...
    def get_cloud_agent_registry(self) -> List[CloudAgentRegistry]: ...
    def get_cloud_agent_registry_by_id(self, registry_id: str) -> Optional[CloudAgentRegistry]: ...
    def get_content_active_censor(self) -> List[ContentActiveCensor]: ...
    def get_content_active_censor_by_id(self, censor_id: str) -> Optional[ContentActiveCensor]: ...
    def get_content_cue(self) -> List[ContentCue]: ...
    def get_content_cue_by_id(self, cue_id: str) -> Optional[ContentCue]: ...
    def get_content_file(self) -> List[ContentFile]: ...
    def get_content_file_by_id(self, file_id: str) -> Optional[ContentFile]: ...
    def get_active_censor(self) -> List[DjmdActiveCensor]: ...
    def get_active_censor_by_id(self, censor_id: str) -> Optional[DjmdActiveCensor]: ...
    def get_album(self) -> List[DjmdAlbum]: ...
    def get_album_by_id(self, album_id: str) -> Optional[DjmdAlbum]: ...
    def get_album_by_name(self, album_name: str) -> Optional[DjmdAlbum]: ...
    def insert_album(
        self, name: str, artist_id: str = None, image_path: str = None, compilation: int = None
    ) -> DjmdAlbum: ...
    def update_album(self, album: DjmdAlbum) -> DjmdAlbum: ...
    def delete_album(self, album_id: str) -> None: ...
    def get_artist(self) -> List[DjmdArtist]: ...
    def get_artist_by_id(self, artist_id: str) -> Optional[DjmdArtist]: ...
    def get_artist_by_name(self, artist_name: str) -> Optional[DjmdArtist]: ...
    def insert_artist(self, name: str) -> DjmdArtist: ...
    def update_artist(self, artist: DjmdArtist) -> DjmdArtist: ...
    def delete_artist(self, artist_id: str) -> None: ...
    def get_category(self) -> List[DjmdCategory]: ...
    def get_category_by_id(self, category_id: str) -> Optional[DjmdCategory]: ...
    def get_color(self) -> List[DjmdColor]: ...
    def get_color_by_id(self, color_id: str) -> Optional[DjmdColor]: ...
    def get_content(self) -> List[DjmdContent]: ...
    def get_content_by_id(self, content_id: str) -> Optional[DjmdContent]: ...
    def get_content_by_path(self, path: str) -> Optional[DjmdContent]: ...
    def get_content_anlz_dir(self, content_id: str) -> str: ...
    def get_content_anlz_paths(self, content_id: str) -> Dict[str, str]: ...
    def insert_content(self, path: str) -> DjmdContent: ...
    def update_content(self, content: DjmdContent) -> None: ...
    def update_content_album(self, content_id: str, name: str) -> None: ...
    def update_content_artist(self, content_id: str, name: str) -> None: ...
    def update_content_remixer(self, content_id: str, name: str) -> None: ...
    def update_content_original_artist(self, content_id: str, name: str) -> None: ...
    def update_content_composer(self, content_id: str, name: str) -> None: ...
    def update_content_genre(self, content_id: str, name: str) -> None: ...
    def update_content_label(self, content_id: str, name: str) -> None: ...
    def update_content_key(self, content_id: str, name: str) -> None: ...
    # def delete_content(self, content_id: str) -> None: ...
    def get_cue(self) -> List[DjmdCue]: ...
    def get_cue_by_id(self, cue_id: str) -> Optional[DjmdCue]: ...
    def get_device(self) -> List[DjmdDevice]: ...
    def get_device_by_id(self, device_id: str) -> Optional[DjmdDevice]: ...
    def get_genre(self) -> List[DjmdGenre]: ...
    def get_genre_by_id(self, genre_id: str) -> Optional[DjmdGenre]: ...
    def get_genre_by_name(self, genre_name: str) -> Optional[DjmdGenre]: ...
    def insert_genre(self, name: str) -> DjmdGenre: ...
    def update_genre(self, genre: DjmdGenre) -> DjmdGenre: ...
    def delete_genre(self, genre_id: str) -> None: ...
    def get_history(self) -> List[DjmdHistory]: ...
    def get_history_by_id(self, history_id: str) -> Optional[DjmdHistory]: ...
    def get_history_songs(self, history_id: str) -> List[DjmdSongHistory]: ...
    def get_history_contents(self, history_id: str) -> List[DjmdContent]: ...
    def get_hot_cue_banklist(self) -> List[DjmdHotCueBanklist]: ...
    def get_hot_cue_banklist_by_id(self, banklist_id: str) -> Optional[DjmdHotCueBanklist]: ...
    def get_hot_cue_banklist_children(self, banklist_id: str) -> List[DjmdHotCueBanklist]: ...
    def get_hot_cue_banklist_songs(self, banklist_id: str) -> List[DjmdSongHotCueBanklist]: ...
    def get_hot_cue_banklist_contents(self, banklist_id: str) -> List[DjmdContent]: ...
    def get_hot_cue_banklist_cues(self, banklist_id: str) -> List[HotCueBanklistCue]: ...
    def get_key(self) -> List[DjmdKey]: ...
    def get_key_by_id(self, key_id: str) -> Optional[DjmdKey]: ...
    def get_key_by_name(self, key_name: str) -> Optional[DjmdKey]: ...
    def insert_key(self, name: str) -> DjmdKey: ...
    def update_key(self, key: DjmdKey) -> DjmdKey: ...
    def delete_key(self, key_id: str) -> None: ...
    def get_label(self) -> List[DjmdLabel]: ...
    def get_label_by_id(self, label_id: str) -> Optional[DjmdLabel]: ...
    def get_label_by_name(self, label_name: str) -> Optional[DjmdLabel]: ...
    def insert_label(self, name: str) -> DjmdLabel: ...
    def update_label(self, label: DjmdLabel) -> DjmdLabel: ...
    def delete_label(self, label_id: str) -> None: ...
    def get_menu_item(self) -> List[DjmdMenuItems]: ...
    def get_menu_item_by_id(self, item_id: str) -> Optional[DjmdMenuItems]: ...
    def get_mixer_param(self) -> List[DjmdMixerParam]: ...
    def get_mixer_param_by_id(self, param_id: str) -> Optional[DjmdMixerParam]: ...
    def get_my_tag(self) -> List[DjmdMyTag]: ...
    def get_my_tag_children(self, tag_id: str) -> List[DjmdMyTag]: ...
    def get_my_tag_by_id(self, tag_id: str) -> Optional[DjmdMyTag]: ...
    def get_my_tag_songs(self, tag_id: str) -> List[DjmdSongMyTag]: ...
    def get_my_tag_contents(self, tag_id: str) -> List[DjmdContent]: ...
    def get_playlist(self) -> List[DjmdPlaylist]: ...
    def get_playlist_tree(self) -> List[DjmdPlaylistTreeItem]: ...
    def get_playlist_children(self, playlist_id: str) -> List[DjmdPlaylist]: ...
    def get_playlist_by_id(self, playlist_id: str) -> Optional[DjmdPlaylist]: ...
    def get_playlist_by_path(self, path: List[str]) -> Optional[DjmdPlaylist]: ...
    def get_playlist_songs(self, playlist_id: str) -> List[DjmdSongPlaylist]: ...
    def get_playlist_contents(self, playlist_id: str) -> List[DjmdContent]: ...
    def get_playlist_song_by_id(self, song_id: str) -> Optional[DjmdSongPlaylist]: ...
    def insert_playlist(
        self,
        name: str,
        attribute: int,
        parent_id: str = None,
        seq: int = None,
        image_path: str = None,
        smart_list: str = None,
    ) -> DjmdPlaylist: ...
    def rename_playlist(self, playlist_id: str, name: str) -> DjmdPlaylist: ...
    def move_playlist(
        self, playlist_id: str, seq: int = None, parent_id: str = None
    ) -> DjmdPlaylist: ...
    def delete_playlist(self, playlist_id: str) -> None: ...
    def insert_playlist_song(
        self, playlist_id: str, content_id: str, seq: int = None
    ) -> DjmdSongPlaylist: ...
    def move_playlist_song(self, song_id: str, seq: int) -> DjmdSongPlaylist: ...
    def delete_playlist_song(self, song_id: str) -> None: ...
    def get_property(self) -> List[DjmdProperty]: ...
    def get_property_by_id(self, property_id: str) -> Optional[DjmdProperty]: ...
    def get_cloud_property(self) -> List[DjmdCloudProperty]: ...
    def get_cloud_property_by_id(self, property_id: str) -> Optional[DjmdCloudProperty]: ...
    def get_recommend_like(self) -> List[DjmdRecommendLike]: ...
    def get_recommend_like_by_id(self, recommend_id: str) -> Optional[DjmdRecommendLike]: ...
    def get_related_tracks(self) -> List[DjmdRelatedTracks]: ...
    def get_related_tracks_children(self, track_id: str) -> List[DjmdRelatedTracks]: ...
    def get_related_tracks_by_id(self, track_id: str) -> Optional[DjmdRelatedTracks]: ...
    def get_related_tracks_songs(self, track_id: str) -> List[DjmdSongRelatedTracks]: ...
    def get_related_tracks_contents(self, track_id: str) -> List[DjmdContent]: ...
    def get_sampler(self) -> List[DjmdSampler]: ...
    def get_sampler_children(self, sampler_id: str) -> List[DjmdSampler]: ...
    def get_sampler_by_id(self, sampler_id: str) -> Optional[DjmdSampler]: ...
    def get_sampler_songs(self, sampler_id: str) -> List[DjmdSongSampler]: ...
    def get_sampler_contents(self, sampler_id: str) -> List[DjmdContent]: ...
    def get_song_tag_list(self) -> List[DjmdSongTagList]: ...
    def get_song_tag_list_by_id(self, tag_list_id: str) -> Optional[DjmdSongTagList]: ...
    def get_sort(self) -> List[DjmdSort]: ...
    def get_sort_by_id(self, sort_id: str) -> Optional[DjmdSort]: ...
    def get_image_file(self) -> List[ImageFile]: ...
    def get_image_file_by_id(self, image_id: str) -> Optional[ImageFile]: ...
    def get_setting_file(self) -> List[SettingFile]: ...
    def get_setting_file_by_id(self, setting_id: str) -> Optional[SettingFile]: ...
    def get_uuid_id_map(self) -> List[UuidIDMap]: ...
    def get_uuid_id_map_by_id(self, uuid_id: str) -> Optional[UuidIDMap]: ...

# ---- ANLZ ----------------------------------------------------------------------------------------

class Beat:
    beat_number: int
    tempo: int
    time: int

class ExtBeat:
    beat_number: int

class BeatGrid:
    beats: List[Beat]

class ExtendedBeatGrid:
    bpm: List[Beat]
    beats: List[ExtBeat]

class Cue:
    hot_cue: int
    status: int
    order_first: int
    order_last: int
    cue_type: int
    time: int
    loop_time: int

class ExtendedCue:
    hot_cue: int
    cue_type: int
    time: int
    loop_time: int
    color: int
    loop_numerator: int
    loop_denominator: int
    comment: str
    hot_cue_color_index: int
    hot_cue_color_red: int
    hot_cue_color_green: int
    hot_cue_color_blue: int

class CueList:
    list_type: int
    cues: List[Cue]

class ExtendedCueList:
    list_type: int
    cues: List[ExtendedCue]

class TinyWaveformColumn:
    height: int

class TinyWaveformPreview:
    data: List[TinyWaveformColumn]

class WaveformColumn:
    height: int
    whiteness: int

class WaveformPreview:
    data: List[WaveformColumn]

class WaveformDetail:
    data: List[WaveformColumn]

class WaveformColorPreviewColumn:
    energy_bottom_half_freq: int
    energy_bottom_third_freq: int
    energy_mid_third_freq: int
    energy_top_third_freq: int

class WaveformColorPreview:
    data: List[WaveformColorPreviewColumn]

class WaveformColorDetailColumn:
    red: int
    green: int
    blue: int
    height: int

class WaveformColorDetail:
    data: List[WaveformColorDetailColumn]

class Waveform3BandColumn:
    mid: int
    high: int
    low: int

class Waveform3BandPreview:
    data: List[Waveform3BandColumn]

class Waveform3BandDetail:
    data: List[Waveform3BandColumn]

class Phrase:
    index: int
    beat: int
    kind: int
    k1: int
    k2: int
    b: int
    beat2: int
    beat3: int
    beat4: int
    k3: int
    fill: int
    beat_fill: int

class SongStructureData:
    mood: int
    end_beat: int
    bank: int
    phrases: List[Phrase]

class PyAnlz:
    def __init__(self, path: str) -> None: ...
    def dump_copy(self, path: str) -> None: ...
    def dump(self) -> None: ...
    def get_tags(self) -> List[str]: ...
    def contains(self, tag_type: str) -> bool: ...
    def get_beat_grid(self) -> Optional[BeatGrid]: ...
    def get_extended_beat_grid(self) -> Optional[ExtendedBeatGrid]: ...
    def get_hot_cues(self) -> Optional[CueList]: ...
    def get_memory_cues(self) -> Optional[CueList]: ...
    def get_extended_hot_cues(self) -> Optional[ExtendedCueList]: ...
    def get_extended_memory_cues(self) -> Optional[ExtendedCueList]: ...
    def get_path(self) -> Optional[str]: ...
    def set_path(self, path: str) -> None: ...
    def get_vbr_data(self) -> Optional[List[int]]: ...
    def get_tiny_waveform_preview(self) -> Optional[TinyWaveformPreview]: ...
    def get_waveform_preview(self) -> Optional[WaveformPreview]: ...
    def get_waveform_detail(self) -> Optional[WaveformDetail]: ...
    def get_waveform_color_preview(self) -> Optional[WaveformColorPreview]: ...
    def get_waveform_color_detail(self) -> Optional[WaveformColorDetail]: ...
    def get_waveform_3band_preview(self) -> Optional[Waveform3BandPreview]: ...
    def get_waveform_3band_detail(self) -> Optional[Waveform3BandDetail]: ...
    def get_song_structure(self) -> Optional[SongStructureData]: ...

# ---- XML -----------------------------------------------------------------------------------------

class Tempo(_RustMutableMapping):
    inizio: float
    bpm: float
    metro: str
    battito: int

    def __init__(self, inizio: float, bpm: float, metro: str, battito: int) -> None: ...

class PositionMark(_RustMutableMapping):
    name: str
    mark_type: int
    num: int
    start: float
    end: Optional[float]

    def __init__(
        self, name: str, mark_type: int, num: int, start: float, end: Optional[float]
    ) -> None: ...

class PlaylistTrack:
    key: str

class Track(_RustMutableMapping):
    trackid: str
    location: str
    tempos: List[Tempo]
    position_marks: List[PositionMark]
    name: Optional[str] = None
    artist: Optional[str] = None
    composer: Optional[str] = None
    album: Optional[str] = None
    grouping: Optional[str] = None
    genre: Optional[str] = None
    kind: Optional[str] = None
    size: Optional[int] = None
    totaltime: Optional[float] = None
    discnumber: Optional[int] = None
    tracknumber: Optional[int] = None
    year: Optional[int] = None
    averagebpm: Optional[float] = None
    datemodified: Optional[datetime] = None
    dateadded: Optional[datetime] = None
    bitrate: Optional[int] = None
    samplerate: Optional[float] = None
    comments: Optional[str] = None
    playcount: Optional[int] = None
    lastplayed: Optional[datetime] = None
    rating: Optional[int] = None
    remixer: Optional[str] = None
    tonality: Optional[str] = None
    label: Optional[str] = None
    mix: Optional[str] = None
    color: Optional[str] = None

    def __init__(self, track_id: str, location: str) -> None: ...

class PlaylistNode:
    name: str
    node_type: int
    key_type: Optional[int]
    tracks: Optional[List[PlaylistTrack]]
    nodes: Optional[List[PlaylistNode]]
    node_path: List[str]

    def __init__(self, node_type: int) -> None: ...
    @classmethod
    def playlist(cls, name: str, key_type: int) -> "PlaylistNode": ...
    @classmethod
    def folder(cls, name: str) -> "PlaylistNode": ...
    def add_node(self, node: "PlaylistNode") -> None: ...
    def new_playlist(self, name: str, key_type: int) -> "PlaylistNode": ...
    def new_folder(self, name: str) -> "PlaylistNode": ...
    def remove_node(self, index: int) -> None: ...
    def clear_nodes(self) -> None: ...
    def get_child(self, index: int) -> "PlaylistNode": ...
    def find_child(self, name: str) -> Optional["PlaylistNode"]: ...
    def get_track(self, index: int) -> PlaylistTrack: ...
    def new_track(self, track_key: str) -> None: ...
    def add_track(self, track: PlaylistTrack) -> None: ...
    def remove_track(self, track_key: str) -> None: ...
    def clear_tracks(self) -> None: ...

class PyRekordboxXml:
    root_playlist: PlaylistNode
    tracks: List[Track]

    def __init__(self, path: str) -> None: ...
    @classmethod
    def load(cls, path: str) -> "PyRekordboxXml": ...
    def to_string(self) -> str: ...
    def dump_copy(self, path: str) -> None: ...
    def dump(self) -> None: ...
    # def get_tracks(self) -> List[Track]: ...
    def get_track(self, index: int) -> Optional[Track]: ...
    def get_track_by_id(self, track_id: str) -> Optional[Track]: ...
    def get_track_by_location(self, location: str) -> Optional[Track]: ...
    def get_track_by_key(self, key: str, key_type: int) -> Optional[Track]: ...
    def add_track(self, track: Track) -> None: ...
    def new_track(self, track_id: str, location: str) -> Track: ...
    def update_track(self, track: Track) -> None: ...
    def remove_track(self, track_id: str) -> None: ...
    def clear_tracks(self) -> None: ...
    def get_playlist_root(self) -> PlaylistNode: ...

# ---- Settings ------------------------------------------------------------------------------------

class PySetting:
    def __init__(self, path: str) -> None: ...
    @classmethod
    def new_mysetting(cls, path: str) -> "PySetting": ...
    @classmethod
    def new_mysetting2(cls, path: str) -> "PySetting": ...
    @classmethod
    def new_djmmysetting(cls, path: str) -> "PySetting": ...
    @classmethod
    def new_devsetting(cls, path: str) -> "PySetting": ...
    @classmethod
    def load(cls, path: str) -> "PySetting": ...
    def dump_copy(self, path: str) -> None: ...
    def dump(self) -> None: ...
    def get_on_air_display(self) -> str: ...
    def set_on_air_display(self, on_air_display: str) -> None: ...
    def get_lcd_brightness(self) -> str: ...
    def set_lcd_brightness(self, lcd_brightness: str) -> None: ...
    def get_quantize(self) -> str: ...
    def set_quantize(self, quantize: str) -> None: ...
    def get_auto_cue_level(self) -> str: ...
    def set_auto_cue_level(self, auto_cue_level: str) -> None: ...
    def get_language(self) -> str: ...
    def set_language(self, language: str) -> None: ...
    def get_jog_ring_brightness(self) -> str: ...
    def set_jog_ring_brightness(self, jog_ring_brightness: str) -> None: ...
    def get_jog_ring_indicator(self) -> str: ...
    def set_jog_ring_indicator(self, jog_ring_indicator: str) -> None: ...
    def get_slip_flashing(self) -> str: ...
    def set_slip_flashing(self, slip_flashing: str) -> None: ...
    def get_disc_slot_illumination(self) -> str: ...
    def set_disc_slot_illumination(self, disc_slot_illumination: str) -> None: ...
    def get_eject_lock(self) -> str: ...
    def set_eject_lock(self, eject_lock: str) -> None: ...
    def get_sync(self) -> str: ...
    def set_sync(self, sync: str) -> None: ...
    def get_play_mode(self) -> str: ...
    def set_play_mode(self, play_mode: str) -> None: ...
    def get_quantize_beat_value(self) -> str: ...
    def set_quantize_beat_value(self, quantize_beat_value: str) -> None: ...
    def get_hotcue_autoload(self) -> str: ...
    def set_hotcue_autoload(self, hotcue_autoload: str) -> None: ...
    def get_hotcue_color(self) -> str: ...
    def set_hotcue_color(self, hotcue_color: str) -> None: ...
    def get_needle_lock(self) -> str: ...
    def set_needle_lock(self, needle_lock: str) -> None: ...
    def get_time_mode(self) -> str: ...
    def set_time_mode(self, time_mode: str) -> None: ...
    def get_jog_mode(self) -> str: ...
    def set_jog_mode(self, jog_mode: str) -> None: ...
    def get_auto_cue(self) -> str: ...
    def set_auto_cue(self, auto_cue: str) -> None: ...
    def get_master_tempo(self) -> str: ...
    def set_master_tempo(self, master_tempo: str) -> None: ...
    def get_tempo_range(self) -> str: ...
    def set_tempo_range(self, tempo_range: str) -> None: ...
    def get_phase_meter(self) -> str: ...
    def set_phase_meter(self, phase_meter: str) -> None: ...
    def get_vinyl_speed_adjust(self) -> str: ...
    def set_vinyl_speed_adjust(self, vinyl_speed_adjust: str) -> None: ...
    def get_jog_display_mode(self) -> str: ...
    def set_jog_display_mode(self, jog_display_mode: str) -> None: ...
    def get_pad_button_brightness(self) -> str: ...
    def set_pad_button_brightness(self, pad_button_brightness: str) -> None: ...
    def get_jog_lcd_brightness(self) -> str: ...
    def set_jog_lcd_brightness(self, jog_lcd_brightness: str) -> None: ...
    def get_waveform_divisions(self) -> str: ...
    def set_waveform_divisions(self, waveform_divisions: str) -> None: ...
    def get_waveform(self) -> str: ...
    def set_waveform(self, waveform: str) -> None: ...
    def get_beat_jump_beat_value(self) -> str: ...
    def set_beat_jump_beat_value(self, beat_jump_beat_value: str) -> None: ...
    def get_channel_fader_curve(self) -> str: ...
    def set_channel_fader_curve(self, channel_fader_curve: str) -> None: ...
    def get_crossfader_curve(self) -> str: ...
    def set_crossfader_curve(self, crossfader_curve: str) -> None: ...
    def get_headphones_pre_eq(self) -> str: ...
    def set_headphones_pre_eq(self, headphones_pre_eq: str) -> None: ...
    def get_headphones_mono_split(self) -> str: ...
    def set_headphones_mono_split(self, headphones_mono_split: str) -> None: ...
    def get_beat_fx_quantize(self) -> str: ...
    def set_beat_fx_quantize(self, beat_fx_quantize: str) -> None: ...
    def get_mic_low_cut(self) -> str: ...
    def set_mic_low_cut(self, mic_low_cut: str) -> None: ...
    def get_talk_over_mode(self) -> str: ...
    def set_talk_over_mode(self, talk_over_mode: str) -> None: ...
    def get_talk_over_level(self) -> str: ...
    def set_talk_over_level(self, talk_over_level: str) -> None: ...
    def get_midi_channel(self) -> str: ...
    def set_midi_channel(self, midi_channel: str) -> None: ...
    def get_midi_button_type(self) -> str: ...
    def set_midi_button_type(self, midi_button_type: str) -> None: ...
    def get_mixer_display_brightness(self) -> str: ...
    def set_mixer_display_brightness(self, mixer_display_brightness: str) -> None: ...
    def get_mixer_indicator_brightness(self) -> str: ...
    def set_mixer_indicator_brightness(self, mixer_indicator_brightness: str) -> None: ...
    def get_channel_fader_curve_long_fader(self) -> str: ...
    def set_channel_fader_curve_long_fader(self, channel_fader_curve_long_fader: str) -> None: ...
    def get_overview_waveform_type(self) -> str: ...
    def set_overview_waveform_type(self, overview_waveform_type: str) -> None: ...
    def get_waveform_color(self) -> str: ...
    def set_waveform_color(self, waveform_color: str) -> None: ...
    def get_key_display_format(self) -> str: ...
    def set_key_display_format(self, key_display_format: str) -> None: ...
    def get_waveform_current_position(self) -> str: ...
    def set_waveform_current_position(self, waveform_current_position: str) -> None: ...
