import os
from glob import glob
from typing import Any
from collections.abc import Callable, Sequence
from concurrent.futures import ThreadPoolExecutor

from dominate.tags import html_tag, td, a, div
from dominate.util import text


def subsetsel(
    content: list[str],
    subset: int | tuple[int, ...] | list[int] | None = None,
) -> list[str]:
    """Subset selection."""
    if subset is None:
        return content
    if isinstance(subset, int):
        return content[:subset]
    if type(subset) is tuple:  # namedtuple is not allowed here
        if len(subset) not in (2, 3):
            raise ValueError('Tuple subset must have length 2 or 3')
        return content[slice(*subset)]
    if isinstance(subset, list):
        return [content[i] for i in subset]
    raise ValueError('Unrecognized subset value')


def parse_pathrep(pathrep: str | tuple[str, str] | list[str] | None) -> tuple[str, str] | None:
    """Parse path replacer."""
    if pathrep is None:
        return None
    if isinstance(pathrep, (tuple, list)):
        pathrep_old = pathrep[0].replace('\\', '/')
        pathrep_new = pathrep[1]
    else:
        pathrep_old = pathrep.replace('\\', '/')
        pathrep_new = ''
    return (pathrep_old, pathrep_new)


def parse_content(
    descriptor: str | list[str],
    subset: int | tuple[int, ...] | list[int] | None = None,
    pathrep: tuple[str, str] | None = None,
    message: str | None = None,
    thumb_func: Callable[[str], str] | None = None,
) -> list[str]:
    """Parse content descriptor."""
    if isinstance(descriptor, list):
        out = descriptor
    elif isinstance(descriptor, str):
        out = sorted(glob(descriptor))
        if len(out) == 0:
            if message is not None:
                print('Warning: %s: no files found matching "%s"' % (message, descriptor))
    else:
        raise ValueError('Invalid type of content descriptor: %s', type(descriptor))
    out = subsetsel(out, subset)
    if not out:
        return out

    if thumb_func is not None:
        with ThreadPoolExecutor() as executor:
            out = list(executor.map(thumb_func, out))

    if pathrep:
        out = [s.replace('\\', '/').replace(pathrep[0], pathrep[1]) for s in out]
    return out


class img_(html_tag):
    """img wrapper that removes empty attributes and supports data-src for lazy loading"""

    tagname = 'img'

    # When True, a provided "src" attribute is moved to "data-src" so that
    # the browser does not automatically start downloading the image. The
    # JavaScript limiter will later copy it back to "src".
    use_data_src = False

    def __init__(self, **kwargs):
        kwargs = {k: v for k, v in kwargs.items() if v}
        if img_.use_data_src and 'src' in kwargs:
            kwargs['data-src'] = kwargs['src']
            del kwargs['src']
        super(img_, self).__init__(**kwargs)


class model_(html_tag):
    tagname = 'model-viewer'


def tda(hrefs: Sequence[str] | None, idx: int, *args, **kwargs) -> Any:
    """td wrapper with optional anchor."""
    if hrefs and idx < len(hrefs) and hrefs[idx]:
        link = a(*args, href=hrefs[idx], target='_blank')
        return td(**kwargs).add(link)  # type: ignore[attr-defined]
    else:
        return td(*args, **kwargs)


def getjs(filename: str) -> str:
    """Return the text content of a JavaScript asset shipped with the package."""
    filedir = os.path.dirname(os.path.realpath(__file__))
    with open(os.path.join(filedir, filename), encoding='utf-8') as fp:
        return fp.read()


def copyright_css() -> str:
    return '.copyright {margin-top: 0.5em; font-size: 85%}'


def copyright_html() -> None:
    with div(cls='copyright'):  # type: ignore[misc]
        text('Generated by')
        a('HTML4Vision', href='https://github.com/mtli/HTML4Vision')
        from . import __version__

        text(' v' + __version__)


def imsize_attrs(imsize: Sequence[int], preserve_aspect: bool) -> dict[str, int | str]:
    if preserve_aspect:
        return {'style': 'max-width:%dpx; max-height:%dpx;' % (imsize[0], imsize[1])}
    else:
        return {'width': imsize[0], 'height': imsize[1]}
