"""Helpers for transforming Arrow tables into HTTP-ready payloads."""
from __future__ import annotations

import datetime as dt
import html
import json
from decimal import Decimal
from typing import Iterable, Mapping, Sequence

import pyarrow as pa

from ..config import Config
from ..core.routes import ParameterSpec, ParameterType
from ..plugins.assets import resolve_image
from .vendor import DEFAULT_CHARTJS_SOURCE


def table_to_records(table: pa.Table) -> list[dict[str, object]]:
    records: list[dict[str, object]] = []
    for row in table.to_pylist():
        converted = {key: _json_friendly(value) for key, value in row.items()}
        records.append(converted)
    return records


def render_table_html(
    table: pa.Table,
    route_metadata: Mapping[str, object] | None,
    config: Config,
    charts: Sequence[Mapping[str, str]] | None = None,
    *,
    postprocess: Mapping[str, object] | None = None,
    watermark: str | None = None,
    params: Sequence[ParameterSpec] | None = None,
    param_values: Mapping[str, object] | None = None,
    format_hint: str | None = None,
    pagination: Mapping[str, object] | None = None,
    rpc_payload: Mapping[str, object] | None = None,
) -> str:
    headers = table.column_names
    records = table_to_records(table)
    table_meta = _merge_view_metadata(route_metadata, "html_t", postprocess)
    params_html = _render_params_ui(
        table_meta,
        params,
        param_values,
        format_hint=format_hint,
        pagination=pagination,
    )
    summary_html = _render_summary_html(
        len(records),
        pagination,
        rpc_payload,
    )
    rpc_html = _render_rpc_payload(rpc_payload)
    rows_html = "".join(
        "<tr>" + "".join(f"<td>{html.escape(str(row.get(col, '')))}</td>" for col in headers) + "</tr>"
        for row in records
    )
    header_html = "".join(f"<th>{html.escape(col)}</th>" for col in headers)
    banners: list[str] = []
    if config.ui.show_http_warning:
        banners.append("<p class='banner warning'>Development mode – HTTP only</p>")
    if config.ui.error_taxonomy_banner:
        banners.append(
            "<p class='banner info'>Errors follow the webbed_duck taxonomy (see docs).</p>"
        )
    chart_html = "".join(item["html"] for item in charts or [])
    watermark_html = _render_watermark_html(watermark)
    styles = (
        "body{font-family:system-ui,sans-serif;margin:1.5rem;}"
        "table{border-collapse:collapse;width:100%;}"
        "th,td{border:1px solid #e5e7eb;padding:0.5rem;text-align:left;}"
        "tr:nth-child(even){background:#f9fafb;}"
        f"{_PARAMS_STYLES}"
        ".banner.warning{color:#b91c1c;}"
        ".banner.info{color:#2563eb;}"
        ".watermark{position:fixed;top:50%;left:50%;transform:translate(-50%,-50%) rotate(-24deg);"
        "font-size:3.5rem;color:rgba(37,99,235,0.12);letter-spacing:0.2rem;pointer-events:none;user-select:none;}"
    )
    return _render_html_document(
        styles=styles,
        watermark_html=watermark_html,
        banners_html="".join(banners),
        chart_html=chart_html,
        params_html=params_html,
        summary_html=summary_html,
        content_html=f"<table><thead><tr>{header_html}</tr></thead><tbody>{rows_html}</tbody></table>",
        rpc_html=rpc_html,
    )


def render_cards_html_with_assets(
    table: pa.Table,
    route_metadata: Mapping[str, object] | None,
    config: Config,
    *,
    charts: Sequence[Mapping[str, str]] | None = None,
    postprocess: Mapping[str, object] | None = None,
    assets: Mapping[str, object] | None = None,
    route_id: str,
    watermark: str | None = None,
    params: Sequence[ParameterSpec] | None = None,
    param_values: Mapping[str, object] | None = None,
    format_hint: str | None = None,
    pagination: Mapping[str, object] | None = None,
    rpc_payload: Mapping[str, object] | None = None,
) -> str:
    metadata = route_metadata or {}
    cards_meta: dict[str, object] = {}
    base_cards = metadata.get("html_c")
    if isinstance(base_cards, Mapping):
        cards_meta.update(base_cards)
    if isinstance(postprocess, Mapping):
        cards_meta.update(postprocess)
    title_col = str(cards_meta.get("title_col") or (table.column_names[0] if table.column_names else "title"))
    image_col = cards_meta.get("image_col")
    meta_cols = cards_meta.get("meta_cols")
    if not isinstance(meta_cols, Sequence):
        meta_cols = [col for col in table.column_names if col not in {title_col, image_col}][:3]

    records = table_to_records(table)
    params_html = _render_params_ui(
        cards_meta,
        params,
        param_values,
        format_hint=format_hint,
        pagination=pagination,
    )
    summary_html = _render_summary_html(
        len(records),
        pagination,
        rpc_payload,
    )
    rpc_html = _render_rpc_payload(rpc_payload)
    getter_name = str(assets.get("image_getter")) if assets and assets.get("image_getter") else None
    base_path = str(assets.get("base_path")) if assets and assets.get("base_path") else None
    cards = []
    for record in records:
        title = html.escape(str(record.get(title_col, "")))
        meta_items = "".join(
            f"<li><span>{html.escape(str(col))}</span>: {html.escape(str(record.get(col, '')))}</li>"
            for col in meta_cols
        )
        image_html = ""
        if image_col and record.get(image_col):
            image_value = str(record[image_col])
            if base_path and not image_value.startswith(("/", "http://", "https://")):
                image_value = f"{base_path.rstrip('/')}/{image_value}"
            resolved = resolve_image(image_value, route_id, getter_name=getter_name)
            image_html = f"<img src='{html.escape(resolved)}' alt='{title}'/>"
        cards.append(
            "<article class='card'>"
            + image_html
            + f"<h3>{title}</h3>"
            + f"<ul>{meta_items}</ul>"
            + "</article>"
        )
    banners: list[str] = []
    if config.ui.show_http_warning:
        banners.append("<p class='banner warning'>Development mode – HTTP only</p>")
    if config.ui.error_taxonomy_banner:
        banners.append("<p class='banner info'>Error taxonomy: user, data, system.</p>")
    chart_html = "".join(item["html"] for item in charts or [])
    watermark_html = _render_watermark_html(watermark)
    styles = (
        "body{font-family:system-ui,sans-serif;margin:1.5rem;}"
        ".cards{display:grid;grid-template-columns:repeat(auto-fit,minmax(220px,1fr));gap:1.25rem;}"
        ".card{border:1px solid #e5e7eb;border-radius:0.5rem;padding:1rem;background:#fff;box-shadow:0 1px 2px rgba(15,23,42,.08);}"
        ".card img{width:100%;height:160px;object-fit:cover;border-radius:0.5rem;}"
        ".card h3{margin:0.5rem 0;font-size:1.1rem;}"
        ".card ul{margin:0;padding-left:1rem;}"
        f"{_PARAMS_STYLES}"
        ".banner.info{color:#2563eb;}"
        ".watermark{position:fixed;top:50%;left:50%;transform:translate(-50%,-50%) rotate(-24deg);"
        "font-size:3.5rem;color:rgba(37,99,235,0.12);letter-spacing:0.2rem;pointer-events:none;user-select:none;}"
    )
    return _render_html_document(
        styles=styles,
        watermark_html=watermark_html,
        banners_html="".join(banners),
        chart_html=chart_html,
        params_html=params_html,
        summary_html=summary_html,
        content_html=f"<section class='cards'>{''.join(cards)}</section>",
        rpc_html=rpc_html,
    )


_PARAMS_STYLES = (
    ".params-bar{margin-bottom:1.25rem;padding:0.85rem 1rem;border:1px solid #e5e7eb;"
    "border-radius:0.75rem;background:#f9fafb;}"
    ".params-form{display:flex;flex-wrap:wrap;gap:0.75rem;align-items:flex-end;}"
    ".param-field{display:flex;flex-direction:column;gap:0.35rem;min-width:12rem;}"
    ".param-field label{font-size:0.85rem;font-weight:600;color:#374151;}"
    ".param-field input,.param-field select{padding:0.45rem 0.6rem;border:1px solid #d1d5db;"
    "border-radius:0.375rem;font:inherit;background:#fff;min-height:2.25rem;}"
    ".param-field select{min-width:10rem;}"
    ".param-help{font-size:0.75rem;color:#6b7280;margin:0;}"
    ".param-actions{display:flex;align-items:center;gap:0.75rem;}"
    ".param-actions button{padding:0.45rem 0.95rem;border-radius:0.375rem;border:1px solid #2563eb;"
    "background:#2563eb;color:#fff;font:inherit;cursor:pointer;}"
    ".param-actions button:hover{background:#1d4ed8;border-color:#1d4ed8;}"
    ".param-actions .reset-link{color:#2563eb;text-decoration:none;font-size:0.9rem;}"
    ".param-actions .reset-link:hover{text-decoration:underline;}"
    ".result-summary{margin:0.5rem 0 1rem 0;color:#374151;font-size:0.9rem;}"
    ".rpc-actions{margin-top:1rem;display:flex;gap:1rem;align-items:center;}"
    ".rpc-actions a{color:#2563eb;text-decoration:none;font-weight:600;}"
    ".rpc-actions a:hover{text-decoration:underline;}"
    ".pagination{margin-top:1rem;}"
    ".pagination a{color:#2563eb;text-decoration:none;font-weight:600;}"
    ".pagination a:hover{text-decoration:underline;}"
)


def _render_watermark_html(watermark: str | None) -> str:
    if not watermark:
        return ""
    return f"<div class='watermark'>{html.escape(watermark)}</div>"


def _render_html_document(
    *,
    styles: str,
    watermark_html: str,
    banners_html: str,
    chart_html: str,
    params_html: str,
    summary_html: str,
    content_html: str,
    rpc_html: str,
) -> str:
    return (
        "<html><head><style>"
        + styles
        + "</style></head><body>"
        + watermark_html
        + banners_html
        + chart_html
        + params_html
        + summary_html
        + content_html
        + rpc_html
        + "</body></html>"
    )


def _merge_view_metadata(
    route_metadata: Mapping[str, object] | None,
    view_key: str,
    postprocess: Mapping[str, object] | None,
) -> dict[str, object]:
    merged: dict[str, object] = {}
    if route_metadata and isinstance(route_metadata.get(view_key), Mapping):
        merged.update(route_metadata[view_key])  # type: ignore[arg-type]
    if isinstance(postprocess, Mapping):
        merged.update(postprocess)
    return merged


def _render_params_ui(
    view_meta: Mapping[str, object] | None,
    params: Sequence[ParameterSpec] | None,
    param_values: Mapping[str, object] | None,
    *,
    format_hint: str | None = None,
    pagination: Mapping[str, object] | None = None,
) -> str:
    if not params:
        return ""
    show: list[str] = []
    if view_meta:
        raw = view_meta.get("show_params")
        if isinstance(raw, str):
            show = [item.strip() for item in raw.split(",") if item.strip()]
        elif isinstance(raw, Sequence) and not isinstance(raw, (str, bytes)):
            show = [str(name) for name in raw]
    if not show:
        return ""
    param_map = {spec.name: spec for spec in params}
    selected_specs = [param_map[name] for name in show if name in param_map]
    if not selected_specs:
        return ""
    values = dict(param_values or {})
    show_set = {spec.name for spec in selected_specs}
    hidden_inputs = []
    format_value = values.get("format") or format_hint
    if format_value:
        hidden_inputs.append(
            "<input type='hidden' name='format' value='"
            + html.escape(_stringify_param_value(format_value))
            + "'/>"
        )
        values.pop("format", None)
    for name, value in values.items():
        if name in show_set:
            continue
        if value in {None, ""}:
            continue
        hidden_inputs.append(
            "<input type='hidden' name='"
            + html.escape(name)
            + "' value='"
            + html.escape(_stringify_param_value(value))
            + "'/>"
        )
    if pagination:
        for key in ("limit", "offset"):
            value = pagination.get(key)
            if value in {None, ""}:
                continue
            hidden_inputs.append(
                "<input type='hidden' name='"
                + html.escape(str(key))
                + "' value='"
                + html.escape(_stringify_param_value(value))
                + "'/>"
            )

    fields: list[str] = []
    for spec in selected_specs:
        control = str(spec.extra.get("ui_control", "")).lower()
        if control not in {"input", "select"}:
            continue
        label = str(spec.extra.get("ui_label") or spec.name.replace("_", " ").title())
        value = values.get(spec.name, spec.default)
        value_str = _stringify_param_value(value)
        field_html = ["<div class='param-field'>"]
        field_html.append(
            "<label for='param-"
            + html.escape(spec.name)
            + "'>"
            + html.escape(label)
            + "</label>"
        )
        if control == "input":
            input_type, extra_attrs = _input_attrs_for_spec(spec)
            placeholder = spec.extra.get("ui_placeholder")
            placeholder_attr = (
                " placeholder='" + html.escape(str(placeholder)) + "'" if placeholder else ""
            )
            field_html.append(
                "<input type='"
                + input_type
                + "' id='param-"
                + html.escape(spec.name)
                + "' name='"
                + html.escape(spec.name)
                + "' value='"
                + html.escape(value_str)
                + "'"
                + extra_attrs
                + placeholder_attr
                + "/>"
            )
        elif control == "select":
            options = _normalize_options(spec.extra.get("options"))
            field_html.append(
                "<select id='param-"
                + html.escape(spec.name)
                + "' name='"
                + html.escape(spec.name)
                + "'>"
            )
            if not options:
                options = [("", "")]
            for opt_value, opt_label in options:
                selected = " selected" if opt_value == value_str else ""
                field_html.append(
                    "<option value='"
                    + html.escape(opt_value)
                    + "'"
                    + selected
                    + ">"
                    + html.escape(opt_label)
                    + "</option>"
                )
            field_html.append("</select>")
        help_text = (
            spec.extra.get("ui_help")
            or spec.extra.get("ui_hint")
            or spec.description
        )
        if help_text:
            field_html.append(
                "<p class='param-help'>" + html.escape(str(help_text)) + "</p>"
            )
        field_html.append("</div>")
        fields.append("".join(field_html))

    if not fields:
        return ""

    form_html = ["<div class='params-bar'><form method='get' class='params-form'>"]
    form_html.extend(hidden_inputs)
    form_html.extend(fields)
    form_html.append(
        "<div class='param-actions'><button type='submit'>Apply</button><a class='reset-link' href='?'>Reset</a></div>"
    )
    form_html.append("</form></div>")
    return "".join(form_html)


def _input_attrs_for_spec(spec: ParameterSpec) -> tuple[str, str]:
    if spec.type is ParameterType.INTEGER:
        return "number", ""
    if spec.type is ParameterType.FLOAT:
        return "number", " step='any'"
    if spec.type is ParameterType.BOOLEAN:
        return "text", ""
    return "text", ""


def _normalize_options(options: object) -> list[tuple[str, str]]:
    normalized: list[tuple[str, str]] = []
    if isinstance(options, Mapping):
        for value, label in options.items():
            normalized.append((
                _stringify_param_value(value),
                str(label) if label is not None else "",
            ))
    elif isinstance(options, Iterable) and not isinstance(options, (str, bytes)):
        for item in options:
            if isinstance(item, Mapping):
                value = item.get("value")
                label = item.get("label", value)
                normalized.append((
                    _stringify_param_value(value),
                    str(label) if label is not None else "",
                ))
            else:
                normalized.append((
                    _stringify_param_value(item),
                    _stringify_param_value(item),
                ))
    return normalized


def _stringify_param_value(value: object) -> str:
    if value is None:
        return ""
    if isinstance(value, bool):
        return "true" if value else "false"
    return str(value)


def _render_summary_html(
    row_count: int,
    pagination: Mapping[str, object] | None,
    rpc_payload: Mapping[str, object] | None,
) -> str:
    total_rows = None
    offset_value = 0
    limit_value = None
    if rpc_payload:
        total_rows = rpc_payload.get("total_rows")
        offset_value = int(rpc_payload.get("offset", 0) or 0)
        limit_raw = rpc_payload.get("limit")
        if limit_raw not in (None, ""):
            try:
                limit_value = int(limit_raw)  # type: ignore[arg-type]
            except (TypeError, ValueError):
                limit_value = None
    if pagination:
        offset_raw = pagination.get("offset")
        limit_raw = pagination.get("limit")
        if offset_raw not in (None, ""):
            try:
                offset_value = int(offset_raw)  # type: ignore[arg-type]
            except (TypeError, ValueError):
                offset_value = offset_value
        if limit_raw not in (None, "") and limit_value is None:
            try:
                limit_value = int(limit_raw)  # type: ignore[arg-type]
            except (TypeError, ValueError):
                limit_value = None
    if total_rows in (None, ""):
        return ""
    start = offset_value + 1 if row_count else offset_value
    end = offset_value + row_count
    total = int(total_rows)
    summary = f"Showing {start:,}–{end:,} of {total:,} rows"
    next_link = None
    if rpc_payload and rpc_payload.get("next_href"):
        next_link = str(rpc_payload["next_href"])
    pagination_html = (
        f"<div class='pagination'><a href='{html.escape(next_link)}'>Next page</a></div>"
        if next_link
        else ""
    )
    return (
        f"<p class='result-summary'>{html.escape(summary)}</p>"
        + pagination_html
    )


def _render_rpc_payload(rpc_payload: Mapping[str, object] | None) -> str:
    if not rpc_payload:
        return ""
    endpoint = rpc_payload.get("endpoint")
    data = {key: value for key, value in rpc_payload.items() if key != "endpoint"}
    if endpoint:
        data["endpoint"] = endpoint
    try:
        payload_json = json.dumps(data, separators=(",", ":"))
    except (TypeError, ValueError):  # pragma: no cover - defensive
        payload_json = "{}"
    link_html = (
        f"<a class='rpc-download' href='{html.escape(str(endpoint))}'>"
        "Download this slice (Arrow)</a>"
        if endpoint
        else ""
    )
    if not link_html and payload_json == "{}":
        return ""
    safe_json = payload_json.replace("</", "<\\/")
    return (
        "<div class='rpc-actions'>"
        + link_html
        + "</div>"
        + "<script type='application/json' id='wd-rpc-config'>"
        + safe_json
        + "</script>"
    )


def build_chartjs_configs(
    table: pa.Table,
    specs: Sequence[Mapping[str, object]],
) -> list[dict[str, object]]:
    configs: list[dict[str, object]] = []
    if not specs:
        return configs

    for index, raw in enumerate(specs):
        if not isinstance(raw, Mapping):
            continue
        chart_type = str(raw.get("type") or "line").strip() or "line"
        chart_id = str(raw.get("id") or f"chart_{index}")
        x_column = str(raw.get("x") or "").strip()

        y_spec = raw.get("y")
        if isinstance(y_spec, Sequence) and not isinstance(y_spec, (str, bytes)):
            y_columns = [str(item) for item in y_spec if str(item)]
        elif isinstance(y_spec, str):
            y_columns = [y_spec]
        else:
            inferred = [name for name in table.column_names if name != x_column]
            y_columns = inferred[:1]

        if not y_columns:
            continue

        labels = _chartjs_labels(table, x_column)
        datasets = _chartjs_datasets(table, y_columns, raw)
        if not datasets:
            continue

        options = {}
        raw_options = raw.get("options")
        if isinstance(raw_options, Mapping):
            options = dict(raw_options)

        title = raw.get("title") or raw.get("label")
        heading = raw.get("heading") or title
        if heading is None:
            heading = chart_id.replace("_", " ").title()

        base_options = {
            "responsive": True,
            "maintainAspectRatio": False,
            "plugins": {
                "legend": {"display": True},
            },
        }
        if title:
            base_options["plugins"]["title"] = {
                "display": True,
                "text": str(title),
            }
        merged_options = _merge_chart_options(base_options, options)

        config = {
            "type": chart_type,
            "data": {
                "labels": labels,
                "datasets": datasets,
            },
            "options": merged_options,
        }
        configs.append({"id": chart_id, "heading": heading, "config": config})

    return configs


def render_chartjs_html(
    charts: Sequence[Mapping[str, object]],
    *,
    config: Config,
    route_id: str,
    route_title: str | None,
    route_metadata: Mapping[str, object] | None,
    postprocess: Mapping[str, object] | None = None,
    default_script_url: str | None = None,
    embed: bool = False,
) -> str:
    meta = _merge_view_metadata(route_metadata, "chart_js", postprocess)
    default_url = default_script_url or DEFAULT_CHARTJS_SOURCE
    cdn_url = str(meta.get("cdn_url") or default_url)
    page_title = str(meta.get("page_title") or route_title or route_id)
    container_class = str(meta.get("container_class") or "wd-chart-grid")
    card_class = str(meta.get("card_class") or "wd-chart-card")
    canvas_height = int(meta.get("canvas_height") or 320)
    empty_message = str(meta.get("empty_message") or "No chart data available.")

    chart_blocks: list[str] = []
    for chart in charts:
        chart_id = str(chart.get("id"))
        heading = chart.get("heading")
        config_payload = chart.get("config")
        if not chart_id or not isinstance(config_payload, Mapping):
            continue
        config_json = _chartjs_config_json(config_payload)
        heading_html = (
            f"<h2>{html.escape(str(heading))}</h2>" if heading else ""
        )
        chart_blocks.append(
            "<section class='"
            + html.escape(card_class)
            + "'>"
            + heading_html
            + "<canvas id='"
            + html.escape(chart_id)
            + "' data-wd-chart='"
            + html.escape(f"{chart_id}-config")
            + "' height='"
            + html.escape(str(canvas_height))
            + "'></canvas>"
            + "<script type='application/json' id='"
            + html.escape(f"{chart_id}-config")
            + "'>"
            + config_json
            + "</script>"
            + "</section>"
        )

    if not chart_blocks:
        chart_blocks.append(
            "<div class='wd-chart-empty'>" + html.escape(empty_message) + "</div>"
        )

    script_tag = (
        "<script src='"
        + html.escape(cdn_url)
        + "' crossorigin='anonymous'></script>"
    )
    boot_script = (
        "<script>(function(){"
        "function init(canvas){"
        "var configId=canvas.getAttribute('data-wd-chart');"
        "if(!configId){return;}"
        "var configEl=document.getElementById(configId);"
        "if(!configEl||canvas.dataset.wdChartLoaded){return;}"
        "canvas.dataset.wdChartLoaded='1';"
        "var cfg=JSON.parse(configEl.textContent||'{}');"
        "var ctx=canvas.getContext('2d');"
        "if(window.Chart){new Chart(ctx,cfg);}"
        "}"
        "function run(){"
        "document.querySelectorAll('canvas[data-wd-chart]').forEach(init);"
        "}"
        "if(document.readyState==='loading'){document.addEventListener('DOMContentLoaded',run);}"
        "else{run();}"
        "})();</script>"
    )

    content = (
        "<div class='"
        + html.escape(container_class)
        + "'>"
        + "".join(chart_blocks)
        + "</div>"
    )

    if embed:
        return script_tag + content + boot_script

    banners: list[str] = []
    if config.ui.show_http_warning:
        banners.append("<p class='banner warning'>Development mode – HTTP only</p>")
    if config.ui.error_taxonomy_banner:
        banners.append(
            "<p class='banner info'>Charts suppress internal error details.</p>"
        )

    styles = (
        "body{font-family:system-ui,sans-serif;margin:1.5rem;background:#f9fafb;"
        "color:#111827;}"
        ".wd-chart-grid{display:grid;gap:1.5rem;grid-template-columns:repeat(auto-fit,minmax(320px,1fr));}"
        ".wd-chart-card{background:#fff;padding:1rem;border-radius:0.75rem;box-shadow:0 1px 2px rgba(15,23,42,.08);}"
        ".wd-chart-card h2{margin:0 0 0.75rem;font-size:1rem;font-weight:600;}"
        "canvas{max-width:100%;}"
        ".wd-chart-empty{padding:1rem;color:#6b7280;font-style:italic;}"
        ".banner.warning{color:#b91c1c;}"
        ".banner.info{color:#2563eb;}"
    )

    return (
        "<!doctype html><html><head><meta charset='utf-8'><title>"
        + html.escape(page_title)
        + "</title><style>"
        + styles
        + "</style></head><body>"
        + "".join(banners)
        + script_tag
        + content
        + boot_script
        + "</body></html>"
    )


def _chartjs_labels(table: pa.Table, column: str) -> list[object]:
    if column and column in table.column_names:
        values = table.column(column).to_pylist()
        return [_json_friendly(value) for value in values]
    return list(range(1, table.num_rows + 1))


def _chartjs_datasets(
    table: pa.Table,
    columns: Sequence[str],
    spec: Mapping[str, object],
) -> list[dict[str, object]]:
    datasets: list[dict[str, object]] = []
    if not columns:
        return datasets

    palette = ["#3b82f6", "#10b981", "#f59e0b", "#ef4444", "#6366f1", "#14b8a6"]
    label_overrides = spec.get("dataset_labels")
    color_overrides = spec.get("colors")
    dataset_overrides = spec.get("dataset_options")

    for idx, column in enumerate(columns):
        if column not in table.column_names:
            continue
        values = table.column(column).to_pylist()
        converted = [_coerce_numeric(value) for value in values]
        if not any(item is not None for item in converted):
            continue

        if len(converted) > table.num_rows:
            converted = converted[: table.num_rows]
        elif len(converted) < table.num_rows:
            converted.extend([None] * (table.num_rows - len(converted)))

        if isinstance(label_overrides, Sequence) and not isinstance(
            label_overrides, (str, bytes)
        ) and idx < len(label_overrides):
            label = str(label_overrides[idx])
        else:
            label = spec.get("label") if len(columns) == 1 else column
            label = str(label)

        if isinstance(color_overrides, Sequence) and not isinstance(
            color_overrides, (str, bytes)
        ) and idx < len(color_overrides):
            color = str(color_overrides[idx])
        else:
            color = palette[idx % len(palette)]

        dataset = {
            "label": label,
            "data": converted,
            "borderColor": color,
            "backgroundColor": color,
        }
        if str(spec.get("type") or "line").strip().lower() in {"line", "radar"}:
            dataset["fill"] = False
            dataset["tension"] = 0.25

        if isinstance(dataset_overrides, Mapping):
            dataset.update(dataset_overrides)
        elif isinstance(dataset_overrides, Sequence) and not isinstance(
            dataset_overrides, (str, bytes)
        ) and idx < len(dataset_overrides):
            override = dataset_overrides[idx]
            if isinstance(override, Mapping):
                dataset.update(override)

        datasets.append(dataset)

    return datasets


def _merge_chart_options(
    base: Mapping[str, object],
    overrides: Mapping[str, object] | None,
) -> dict[str, object]:
    merged = dict(base)
    if not overrides:
        return merged
    for key, value in overrides.items():
        if (
            key in merged
            and isinstance(merged[key], Mapping)
            and isinstance(value, Mapping)
        ):
            merged[key] = _merge_chart_options(merged[key], value)
        else:
            merged[key] = value
    return merged


def _chartjs_config_json(config: Mapping[str, object]) -> str:
    try:
        payload = json.dumps(
            config,
            default=_json_friendly,
            separators=(",", ":"),
        )
    except (TypeError, ValueError):  # pragma: no cover - defensive fallback
        payload = "{}"
    return payload.replace("</", "<\\/")


def _coerce_numeric(value: object) -> float | None:
    if value is None:
        return None
    if isinstance(value, bool):
        return 1.0 if value else 0.0
    if isinstance(value, (int, float)):
        return float(value)
    if isinstance(value, Decimal):
        return float(value)
    if isinstance(value, dt.datetime):
        return value.timestamp()
    if isinstance(value, dt.date):
        return float(dt.datetime.combine(value, dt.time.min).timestamp())
    try:
        text = str(value)
        if not text:
            return None
        return float(text)
    except (TypeError, ValueError):
        return None


def render_feed_html(
    table: pa.Table,
    route_metadata: Mapping[str, object] | None,
    config: Config,
    *,
    postprocess: Mapping[str, object] | None = None,
) -> str:
    metadata = route_metadata or {}
    feed_meta = metadata.get("feed", {})
    if not isinstance(feed_meta, Mapping):
        feed_meta = {}
    if isinstance(postprocess, Mapping):
        merged = dict(feed_meta)
        merged.update(postprocess)
        feed_meta = merged
    ts_col = str(feed_meta.get("timestamp_col") or (table.column_names[0] if table.column_names else "timestamp"))
    title_col = str(feed_meta.get("title_col") or (table.column_names[1] if len(table.column_names) > 1 else "title"))
    summary_col = feed_meta.get("summary_col")

    records = table_to_records(table)
    groups: dict[str, list[str]] = {"Today": [], "Yesterday": [], "Earlier": []}
    now = dt.datetime.now(dt.timezone.utc)
    for record in records:
        ts_value = record.get(ts_col)
        if isinstance(ts_value, str):
            try:
                ts = dt.datetime.fromisoformat(ts_value)
            except ValueError:
                ts = now
        elif isinstance(ts_value, dt.datetime):
            ts = ts_value
        else:
            ts = now
        if ts.tzinfo is None:
            ts = ts.replace(tzinfo=dt.timezone.utc)
        delta = now.date() - ts.astimezone(dt.timezone.utc).date()
        if delta.days == 0:
            bucket = "Today"
        elif delta.days == 1:
            bucket = "Yesterday"
        else:
            bucket = "Earlier"
        title = html.escape(str(record.get(title_col, "")))
        summary = html.escape(str(record.get(summary_col, ""))) if summary_col else ""
        entry = f"<article><h4>{title}</h4><p>{summary}</p><time>{ts.isoformat()}</time></article>"
        groups[bucket].append(entry)

    sections = []
    for bucket, entries in groups.items():
        if not entries:
            continue
        sections.append(f"<section><h3>{bucket}</h3>{''.join(entries)}</section>")
    taxonomy = ""
    if config.ui.error_taxonomy_banner:
        taxonomy = "<aside class='banner info'>Feeds suppress sensitive system errors.</aside>"
    return (
        "<html><head><style>"
        "body{font-family:system-ui,sans-serif;margin:1.5rem;}"
        "section{margin-bottom:1.5rem;}"
        "h3{color:#111827;}"
        "article{padding:0.75rem 0;border-bottom:1px solid #e5e7eb;}"
        "time{display:block;color:#6b7280;font-size:0.875rem;}"
        ".banner.info{color:#2563eb;}"
        "</style></head><body>"
        + taxonomy
        + "".join(sections)
        + "</body></html>"
    )


def _json_friendly(value: object) -> object:
    if isinstance(value, (dt.date, dt.datetime)):
        return value.isoformat()
    return value


__all__ = [
    "render_cards_html_with_assets",
    "render_feed_html",
    "render_chartjs_html",
    "build_chartjs_configs",
    "render_table_html",
    "table_to_records",
]
