# ses-cloudwatch

Small CDK construct that routes Amazon SES sending events to a CloudWatch LogGroup through EventBridge. It creates:

* CloudWatch LogGroup (optionally named)
* SES Configuration Set (optionally named)
* EventBridge Rule (matches all `aws.ses` events)
* Configuration Set Event Destination that sends `SEND` events to the default EventBridge bus which the rule targets to the LogGroup

## Install

Add as a dependency to your CDK app (peer deps `aws-cdk-lib` and `constructs` must already be present at compatible versions):

```bash
npm install ses-cloudwatch
```

## Usage

```python
import { Stack } from 'aws-cdk-lib';
import { SesCloudWatch } from 'ses-cloudwatch';

class MyStack extends Stack {
	constructor(scope: Construct, id: string) {
		super(scope, id);

		new SesCloudWatch(this, 'SesEvents');
	}
}
```

### With custom names

```python
new SesCloudWatch(this, 'SesEvents', {
	logGroupName: 'my-ses-event-logs',
	configurationSetName: 'marketing-emails',
	eventRuleName: 'marketing-emails-events',
});
```

### Capturing additional SES events

By default only `SEND` events are forwarded. Supply the `events` prop to include more:

```python
import { aws_ses as ses } from 'aws-cdk-lib';

new SesCloudWatch(this, 'SesEvents', {
	events: [
		ses.EmailSendingEvent.SEND,
		ses.EmailSendingEvent.REJECT,
		ses.EmailSendingEvent.DELIVERY,
	],
});
```

## Props

| Name | Type | Default | Description |
| ---- | ---- | ------- | ----------- |
| `logGroupName` | `string?` | auto-generated by CloudFormation | Explicit name for the CloudWatch LogGroup that stores SES events. |
| `configurationSetName` | `string?` | `<stackName>-ses-config-set` | Name for the SES Configuration Set whose events are logged. |
| `eventRuleName` | `string?` | `<stackName>-ses-event-rule` | Name for the EventBridge Rule that matches SES events. |
| `events` | `ses.EmailSendingEvent[]?` | `[ses.EmailSendingEvent.SEND]` | Set of SES sending event types to capture. |

## Testing

Run the Jest tests (auto-generated by projen) which validate both default and custom prop behavior:

```bash
npm test
```

## License

Apache-2.0
