"""Built-in error handlers."""
import traceback
from functools import wraps
from typing import Callable, Optional

from .exceptions import HTTPError
from .request import Request
from .response import Response


def error_to_html(req: Request, res: Response, exc: HTTPError):
    """Convert an exception to an HTML response.

    The response contains a `<h1>` tag with the error's `title` and,
    if provided, a `<p>` tag with the error's `detail`.

    # Example

    ```html
    <h1>403 Forbidden</h1>
    <p>You do not have the permissions to perform this operation.</p>
    ```
    """
    res.status_code = exc.status_code
    html = f"<h1>{exc.title}</h1>"
    if exc.detail:
        html += f"\n<p>{exc.detail}</p>"
    res.html = html


def error_to_media(req: Request, res: Response, exc: HTTPError):
    """Convert an exception to a media response.

    The response contains the following items:

    - `error`: the error's `title`
    - `status`: the error's `status_code`
    - `detail`: the error's `detail` (if provided)

    # Example

    ```json
    {
        "error": "403 Forbidden",
        "status": 403,
        "detail": "You do not have the permissions to perform this operation."
    }
    ```
    """
    res.status_code = exc.status_code
    media = {"error": exc.title, "status": exc.status_code}
    if exc.detail:
        media["detail"] = exc.detail
    res.media = media


def error_to_text(req: Request, res: Response, exc: HTTPError):
    """Convert an exception to a plain text response.

    The response contains a line with the error's `title` and, if provided,
    a line for the error's `detail`.

    # Example
    ```
    403 Forbidden
    You do not have the permissions to perform this operation.
    ```
    """
    res.status_code = exc.status_code
    text = exc.title
    if exc.detail:
        text += f"\n{exc.detail}"
    res.text = text


ErrorHandler = Callable[[Request, Response, Exception], None]


async def _to_res(
    req: Request, exc: Exception, error_handler: ErrorHandler, **kwargs
) -> Response:
    if isinstance(exc, HTTPError):
        res = Response(req, **kwargs)
        error_handler(req, res, exc)
        if exc.status_code == 500:
            traceback.print_exc()
    else:
        res = await _to_res(req, HTTPError(500), error_handler, **kwargs)
    return res


def convert_exception_to_response(
    dispatch, error_handler: Optional[ErrorHandler] = None, **kwargs
):
    # Wrap call to `dispatch()` to always return an HTTP response.
    if error_handler is None:
        error_handler = error_to_text

    @wraps(dispatch)
    async def inner(req: Request) -> Response:
        try:
            res = await dispatch(req)
        except Exception as exc:
            res = await _to_res(req, exc, error_handler, **kwargs)
        return res

    return inner
