import pandas as pd
import numpy as np
import SimpleITK as sitk

class Helper:
    def __init__(self):
        pass
    def resample_to_isotropic(self, image, new_spacing=[2.0, 2.0, 2.0], interpolator=sitk.sitkLinear):
        original_spacing = image.GetSpacing()
        original_size = image.GetSize()
        original_direction = image.GetDirection()
        original_origin = image.GetOrigin()
    
        new_size = [
            int(round(original_size[0] * original_spacing[0] / new_spacing[0])),
            int(round(original_size[1] * original_spacing[1] / new_spacing[1])),
            int(round(original_size[2] * original_spacing[2] / new_spacing[2]))
        ]
    
        resampler = sitk.ResampleImageFilter()
        resampler.SetOutputSpacing(new_spacing)
        resampler.SetSize(new_size)
        resampler.SetOutputDirection(original_direction)
        resampler.SetOutputOrigin(original_origin)
        resampler.SetInterpolator(interpolator)
        resampler.SetTransform(sitk.Transform())
    
        resampled_image = resampler.Execute(image)
        return resampled_image
    def split_dataframe(self,df, n, random_seed=None):
        """
        将 DataFrame 按固定随机选择分成两个子集
        
        参数：
        df: 原始 DataFrame
        n: 需要抽取的行数
        random_seed: 随机种子（确保可复现性）
        
        返回：
        df1: 包含随机选取的n行的DataFrame
        df2: 包含剩余行的DataFrame
        """
        # 参数有效性检查
        if not isinstance(n, int) or n < 0:
            raise ValueError("n 必须是正整数")
        if n > len(df):
            raise ValueError("n 不能超过DataFrame的总行数")
        
        df1 = df.sample(n=n, random_state=random_seed)
        
        df2 = df.drop(df1.index)
        
        df1 = df1.reset_index(drop=True)
        df2 = df2.reset_index(drop=True)
        
        return df1, df2
    def detailed_nan_inspection(self,df):
        nan_report = df.isna().sum().to_frame('NaN Count')
        inf_report = pd.DataFrame(index=df.columns, columns=['Inf Count'])
        
        # 遍历所有列检测无限值
        for col in df.columns:
            if pd.api.types.is_numeric_dtype(df[col]):
                inf_report.loc[col] = np.isinf(df[col]).sum()
            else:
                inf_report.loc[col] = 0  # 非数值列无inf
        
        # 合并报告并筛选问题列
        report = nan_report.join(inf_report)
        problem_cols = report[(report['NaN Count']>0) | (report['Inf Count']>0)]
        
        # 可视化显示
        if not problem_cols.empty:
            display(problem_cols.style.bar(color='#FFA07A'))
            sample_info = {}
            for col in problem_cols.index:
                bad_samples = df[col][df[col].isna() | np.isinf(df[col])].index[:3]  # 显示前3个异常样本
                sample_info[col] = f"异常样本ID: {list(bad_samples)}"
            return {"msg":"error"}
        else:
            return {"msg":"correct"}
    def get_mpum_categories(self):
        return {"0": "background", "1": "left_to_annotate", "2": "muscles", "3": "fat", "4": "abdominal_tissue", "5": "mediastinal_tissue", "6": "esophagus", "7": "stomach", "8": "small_bowel", "9": "duodenum", "10": "colon", "11": "rectum", "12": "gallbladder", "13": "liver", "14": "pancreas", "15": "kidney_left", "16": "kidney_right", "17": "urinary_bladder", "18": "gonads", "19": "prostate", "20": "uterocervix", "21": "uterus", "22": "breast_left", "23": "breast_right", "24": "spinal_canal", "25": "brain", "26": "spleen", "27": "adrenal_gland_left", "28": "adrenal_gland_right", "29": "thyroid_left", "30": "thyroid_right", "31": "thymus", "32": "gluteus_maximus_left", "33": "gluteus_maximus_right", "34": "gluteus_medius_left", "35": "gluteus_medius_right", "36": "gluteus_minimus_left", "37": "gluteus_minimus_right", "38": "iliopsoas_left", "39": "iliopsoas_right", "40": "autochthon_left", "41": "autochthon_right", "42": "skin", "43": "vertebrae_C1", "44": "vertebrae_C2", "45": "vertebrae_C3", "46": "vertebrae_C4", "47": "vertebrae_C5", "48": "vertebrae_C6", "49": "vertebrae_C7", "50": "vertebrae_T1", "51": "vertebrae_T2", "52": "vertebrae_T3", "53": "vertebrae_T4", "54": "vertebrae_T5", "55": "vertebrae_T6", "56": "vertebrae_T7", "57": "vertebrae_T8", "58": "vertebrae_T9", "59": "vertebrae_T10", "60": "vertebrae_T11", "61": "vertebrae_T12", "62": "vertebrae_L1", "63": "vertebrae_L2", "64": "vertebrae_L3", "65": "vertebrae_L4", "66": "vertebrae_L5", "67": "rib_left_1", "68": "rib_right_1", "69": "rib_left_2", "70": "rib_right_2", "71": "rib_left_3", "72": "rib_right_3", "73": "rib_left_4", "74": "rib_right_4", "75": "rib_left_5", "76": "rib_right_5", "77": "rib_left_6", "78": "rib_right_6", "79": "rib_left_7", "80": "rib_right_7", "81": "rib_left_8", "82": "rib_right_8", "83": "rib_left_9", "84": "rib_right_9", "85": "rib_left_10", "86": "rib_right_10", "87": "rib_left_11", "88": "rib_right_11", "89": "rib_left_12", "90": "rib_right_12", "91": "rib_cartilage", "92": "sternum", "93": "clavicle_left", "94": "clavicle_right", "95": "scapula_left", "96": "scapula_right", "97": "humerus_left", "98": "humerus_right", "99": "skull", "100": "hip_left", "101": "hip_right", "102": "sacrum", "103": "femur_left", "104": "femur_right", "105": "heart", "106": "heart_atrium_left", "107": "heart_tissue", "108": "heart_atrium_right", "109": "heart_myocardium", "110": "heart_ventricle_left", "111": "heart_ventricle_right", "112": "iliac_artery_left", "113": "iliac_artery_right", "114": "aorta", "115": "iliac_vena_left", "116": "iliac_vena_right", "117": "inferior_vena_cava", "118": "portal_vein_and_splenic_vein", "119": "celiac_trunk", "120": "lung_lower_lobe_left", "121": "lung_upper_lobe_left", "122": "lung_lower_lobe_right", "123": "lung_middle_lobe_right", "124": "lung_upper_lobe_right", "125": "bronchie", "126": "trachea", "127": "pulmonary_artery", "128": "cheek_left", "129": "cheek_right", "130": "eyeball_left", "131": "eyeball_right", "132": "R-Hippocampus", "133": "L-Hippocampus", "134": "R-Amygdala", "135": "L-Amygdala", "136": "R-Anterior-temporal-lobe-medial-part", "137": "L-Anterior-temporal-lobe-medial-part", "138": "R-Anterior-temporal-lobe-lateral-part", "139": "L-Anterior-temporal-lobe-lateral-part", "140": "R-Parahippocampal-and-ambient-gyri", "141": "L-Parahippocampal-and-ambient-gyri", "142": "R-Superior-temporal-gyrus-posterior-part", "143": "L-Superior-temporal-gyrus-posterior-part", "144": "R-Middle and inferior temporal gyrus", "145": "L-Middle and inferior temporal gyrus", "146": "R-Fusiform gyrus", "147": "L-Fusiform gyrus", "148": "R-Cerebellum", "149": "L-Cerebellum", "150": "Brainstem", "151": "L-Insula", "152": "R-Insula", "153": "L-Lateral remainder of occipital lobe", "154": "R-Lateral remainder of occipital lobe", "155": "L-Cingulate gyrus gyrus cinguli anterior part", "156": "R-Cingulate gyrus gyrus cinguli anterior part", "157": "L-Cingulate gyrus gyrus cinguli posterior part", "158": "R-Cingulate gyrus gyrus cinguli posterior part", "159": "L-Middle frontal gyrus", "160": "R-Middle frontal gyrus", "161": "L-Posterior temporal lobe", "162": "R-Posterior temporal lobe", "163": "L-Inferiolateral remainder of parietal lobe", "164": "R-Inferiolateral remainder of parietal lobe", "165": "L-Caudate nucleus", "166": "R-Caudate nucleus", "167": "L-Nucleus accumbens", "168": "R-Nucleus accumbens", "169": "L-Putamen", "170": "R-Putamen", "171": "L-Thalamus", "172": "R-Thalamus", "173": "L-Pallidum", "174": "R-Pallidum", "175": "Corpus callosum", "176": "R-Lateral ventricle excluding temporal horn", "177": "L-Lateral ventricle excluding temporal horn", "178": "R-Lateral ventricle, temporal horn", "179": "L-Lateral ventricle, temporal horn", "180": "Third ventricle", "181": "L-Precentral gyrus", "182": "R-Precentral gyrus", "183": "L-Straight gyrus", "184": "R-Straight gyrus", "185": "L-Anterior orbital gyrus", "186": "R-Anterior orbital gyrus", "187": "L-Inferior frontal gyrus", "188": "R-Inferior frontal gyrus", "189": "L-Superior frontal gyrus", "190": "R-Superior frontal gyrus", "191": "L-Postcentral gyrus", "192": "R-Postcentral gyrus", "193": "L-Superior parietal gyrus", "194": "R-Superior parietal gyrus", "195": "L-Lingual gyrus", "196": "R-Lingual gyrus", "197": "L-Cuneus", "198": "R-Cuneus", "199": "L-Medial orbital gyrus", "200": "R-Medial orbital gyrus", "201": "L-Lateral orbital gyrus", "202": "R-Lateral orbital gyrus", "203": "L-Posterior orbital gyrus", "204": "R-Posterior orbital gyrus", "205": "L-Substantia nigra", "206": "R-Substantia nigra", "207": "L-Subgenual frontal cortex", "208": "R-Subgenual frontal cortex", "209": "L-Subcallosal area", "210": "R-Subcallosal area", "211": "L-Pre-subgenual frontal cortex", "212": "R-Pre-subgenual frontal cortex", "213": "L-Superior temporal gyrus anterior part", "214": "R-Superior temporal gyrus anterior part"}