use std::{convert::Infallible, ops::Deref, str::FromStr, sync::Arc};

use pyo3::{
    pyclass, pymethods,
    types::{PyAnyMethods, PyString},
    Bound, FromPyObject, IntoPyObject, Python,
};
use strum::{EnumString, VariantArray, VariantNames};
use web_time::Instant;
#[cfg(any(
    target_os = "windows",
    target_os = "macos",
    target_os = "linux",
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd"
))]
use winit::platform::scancode::PhysicalKeyExtScancode;
use winit::{event as winit_event, keyboard::Key};

use crate::{
    time::Timestamp,
    visual::{geometry::Size, window::Window},
};

// pub mod video;

/// A mouse button.
#[derive(Debug, Clone, PartialEq)]
#[pyclass]
pub enum MouseButton {
    /// The left mouse button.
    Left(),
    /// The right mouse button.
    Right(),
    /// The middle mouse button.
    Middle(),
    /// The forward mouse button.
    Forward(),
    /// The back mouse button.
    Back(),
    /// An additional mouse button with the given index.
    Other(u16),
}

#[derive(Debug, Clone, enum_fields::EnumFields, strum::EnumDiscriminants)]
#[pyclass]
#[strum_discriminants(
    name(EventKind),
    strum(serialize_all = "snake_case"),
    derive(EnumString, VariantNames, strum::Display)
)]
pub enum Event {
    /// A keypress event. This is triggered when a key is pressed.
    KeyPress {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// String representation of the key that was pressed.
        key: String,
        /// KeyCode of the key that was pressed.
        code: u32,
    },
    /// A key release event. This is triggered when a key is released.
    KeyRelease {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// String representation of the key that was released.
        key: String,
        /// KeyCode of the key that was released.
        code: u32,
    },

    /// A mouse button press event. This is triggered when a mouse button is
    /// pressed.
    MouseButtonPress {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The button that was pressed.
        button: MouseButton,
        /// The position of the mouse cursor when the button was pressed.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
    },

    /// A mouse button release event. This is triggered when a mouse button is
    /// released.
    MouseButtonRelease {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The button that was released.
        button: MouseButton,
        /// The position of the mouse cursor when the button was released.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// A touch start event. This is triggered when a touch screen is touched.
    TouchStart {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the touch.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
        /// The id of the touch (if available).
        id: Option<u64>,
    },
    /// A touch move event. This is triggered when a touch screen is moved.
    TouchMove {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the touch.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
        /// The id of the touch (if available).
        id: Option<u64>,
    },
    /// A touch end event. This is triggered when a touch screen is
    /// released.
    TouchEnd {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the touch.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
        /// The id of the touch (if available).
        id: Option<u64>,
    },
    /// A touch cancel event. This is triggered when a touch screen is
    /// cancelled.
    TouchCancel {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the touch.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
        /// The id of the touch (if available).
        id: Option<u64>,
    },
    /// The window has lost focus.
    FocusGained {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// The window has gained focus.
    FocusLost {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// The mouse cursor was moved.
    CursorMoved {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the cursor.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// The mouse cursor was entered into the window.
    CursorEntered {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The position of the cursor when it entered the window.
        position: (f32, f32),
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// The mouse cursor was exited from the window.
    CursorExited {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// A pressure-sensitive touchpad was pressed (if available).
    TouchpadPress {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The pressure of the touch.
        pressure: f32,
        /// The level of the touch.
        stage: i64,
        /// The Window that the event was triggered on.
        window: Window,
    },
    /// The mouse wheel was scrolled (or the equivalent touchpad gesture).
    MouseWheel {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The amount of horizontal scrolling.
        horizontal: f32,
        /// The amount of vertical scrolling.
        vertical: f32,
    },
    /// Onset event.
    Onset {
        /// Timestamp of the event.
        timestamp: Timestamp,
        // /// If the provided timestamp has been corrected/estimated.
        // corrected: bool,
        // /// If corrected, the original timestamp before correction.
        // original_timestamp: Option<Timestamp>,
    },
    /// Offset event.
    Offset {
        /// Timestamp of the event.
        timestamp: Timestamp,
    },
    /// Any other event. The string contains the name of the event.
    Other {
        /// Timestamp of the event.
        timestamp: Timestamp,
        /// The name of the event.
        name: String,
    },
}

// Implements convenience methods for Event.
impl Event {
    /// Returns true if this element represents a press of the given key.
    pub fn key_pressed(&self, key: &str) -> bool {
        matches!(&self, Self::KeyPress { key: k, .. } if k == key)
    }

    /// Returns true if this element represents a release of the given key.
    pub fn key_released(&self, key: &str) -> bool {
        matches!(&self, Self::KeyRelease { key: k, .. } if k == key)
    }

    /// Returns true if this element represents a press of the given mouse
    /// button.
    pub fn mouse_button_pressed(&self, button_a: MouseButton) -> bool {
        matches!(&self, Self::MouseButtonPress { button, .. } if button_a == *button)
    }

    /// Returns true if this element represents a release of the given mouse
    /// button.
    pub fn mouse_button_released(&self, button_a: MouseButton) -> bool {
        matches!(&self, Self::MouseButtonRelease { button, .. } if button_a == *button)
    }

    /// Returns true if this element represents a release of the left mouse button.
    pub fn left_mouse_button_released(&self) -> bool {
        matches!(
            &self,
            Self::MouseButtonRelease {
                button: MouseButton::Left(),
                ..
            }
        )
    }

    /// Returns true if this element represents a press of the left mouse button.
    pub fn left_mouse_button_pressed(&self) -> bool {
        matches!(
            &self,
            Self::MouseButtonPress {
                button: MouseButton::Left(),
                ..
            }
        )
    }

    /// Returns true if this element represents a press of the right mouse button.
    pub fn right_mouse_button_pressed(&self) -> bool {
        matches!(
            &self,
            Self::MouseButtonPress {
                button: MouseButton::Right(),
                ..
            }
        )
    }

    /// Returns true if this element represents a release of the right mouse button.
    pub fn right_mouse_button_released(&self) -> bool {
        matches!(
            &self,
            Self::MouseButtonRelease {
                button: MouseButton::Right(),
                ..
            }
        )
    }

    /// Returns the kind of this event.
    pub fn kind(&self) -> EventKind {
        self.into()
    }
}

#[pymethods]
impl Event {
    #[getter]
    #[pyo3(name = "position")]
    fn py_position(&self) -> Option<(f32, f32)> {
        self.position().cloned()
    }

    #[getter]
    #[pyo3(name = "window")]
    fn py_window(&self) -> Option<Window> {
        self.window().cloned()
    }

    #[getter]
    #[pyo3(name = "key")]
    fn py_key(&self) -> Option<String> {
        self.key().cloned()
    }

    #[getter]
    #[pyo3(name = "id")]
    fn py_id(&self) -> Option<u64> {
        self.id().cloned().flatten()
    }

    #[getter]
    #[pyo3(name = "pressure")]
    fn py_pressure(&self) -> Option<f32> {
        self.pressure().cloned()
    }

    #[getter]
    #[pyo3(name = "stage")]
    fn py_stage(&self) -> Option<i64> {
        self.stage().cloned()
    }

    #[getter]
    #[pyo3(name = "name")]
    fn py_name(&self) -> Option<String> {
        self.name().cloned()
    }

    #[getter]
    #[pyo3(name = "kind")]
    fn py_kind(&self) -> EventKind {
        self.kind()
    }
}

// Custom conversion from winit events to InputEvents.
pub(crate) trait EventTryFrom<T>: Sized {
    type Error;

    fn try_from_winit(value: T, window: &Window) -> Result<Self, Self::Error>;
}

/// Convert a winit WindowEvent to an InputEvent.
impl EventTryFrom<winit_event::WindowEvent> for Event {
    type Error = &'static str;

    fn try_from_winit(event: winit_event::WindowEvent, window: &Window) -> Result<Self, Self::Error> {
        let timestamp = Instant::now();
        let data = match event {
            // match keyboad events
            winit_event::WindowEvent::KeyboardInput {
                device_id: _, event, ..
            } => {
                let key = event.logical_key;

                let key_str = match key {
                    Key::Named(key) => Some(format!("{:?}", key).to_string()),
                    Key::Character(key) => Some(key.to_string()),
                    _ => None,
                };

                let key_str = match key_str {
                    Some(key) => key,
                    None => return Err("Failed to convert key to string"),
                };

                let key_code = u32::default();

                #[cfg(any(
                    target_os = "windows",
                    target_os = "macos",
                    target_os = "linux",
                    target_os = "freebsd",
                    target_os = "dragonfly",
                    target_os = "openbsd",
                    target_os = "netbsd"
                ))]
                let key_code = event.physical_key.to_scancode().unwrap_or_default();

                match event.state {
                    winit_event::ElementState::Pressed => Event::KeyPress {
                        timestamp: timestamp.into(),
                        key: key_str.to_string(),
                        code: key_code,
                    },
                    winit_event::ElementState::Released => Event::KeyRelease {
                        timestamp: timestamp.into(),
                        key: key_str.to_string(),
                        code: key_code,
                    },
                }
            }
            // match mouse button events
            winit_event::WindowEvent::PointerButton {
                device_id,
                state,
                position,
                primary,
                button,
            } => {
                let button = match button {
                    winit_event::ButtonSource::Mouse(winit_event::MouseButton::Left) => MouseButton::Left(),
                    winit_event::ButtonSource::Mouse(winit_event::MouseButton::Right) => MouseButton::Right(),
                    winit_event::ButtonSource::Mouse(winit_event::MouseButton::Middle) => MouseButton::Middle(),
                    winit_event::ButtonSource::Mouse(winit_event::MouseButton::Back) => MouseButton::Back(),
                    winit_event::ButtonSource::Mouse(winit_event::MouseButton::Forward) => MouseButton::Forward(),
                    _ => MouseButton::Other(0), // Default case, should not happen
                };

                let position = {
                    // move by x_origin and y_origin
                    let window_state = window.state.lock().unwrap();
                    let window_state = window_state.as_ref().unwrap();
                    let window_size = window_state.size;
                    let pos = (position.x as f32, position.y as f32);

                    (
                        pos.0 - (window_size.width as f32 / 2.0),
                        pos.1 - (window_size.height as f32 / 2.0),
                    )
                };

                match state {
                    winit_event::ElementState::Pressed => Event::MouseButtonPress {
                        timestamp: timestamp.into(),
                        button,
                        position,
                        window: window.clone(),
                    },
                    winit_event::ElementState::Released => Event::MouseButtonRelease {
                        timestamp: timestamp.into(),
                        button,
                        position,
                        window: window.clone(),
                    },
                }
            }
            winit_event::WindowEvent::PointerEntered {
                device_id,
                position,
                primary,
                kind,
            } => {
                //  let position = (Size::Pixels(position.x) - Size::ScreenWidth(0.5), Size::Pixels(-position.y) + Size::ScreenHeight(0.5));
                let position = {
                    // move by x_origin and y_origin
                    let window_state = window.state.lock().unwrap();
                    let window_state = window_state.as_ref().unwrap();
                    let window_size = window_state.size;
                    let pos = (position.x as f32, position.y as f32);

                    (
                        pos.0 - (window_size.width as f32 / 2.0),
                        pos.1 - (window_size.height as f32 / 2.0),
                    )
                };

                match kind {
                    // this is a touch even
                    winit_event::PointerKind::Touch(finger_id) => Event::TouchStart {
                        timestamp: timestamp.into(),
                        position,
                        window: window.clone(),
                        id: Some(finger_id.into_raw() as u64),
                    },
                    // this is a mouse event
                    _ => Event::CursorEntered {
                        timestamp: timestamp.into(),
                        position: position,
                        window: window.clone(),
                    },
                }
            }
            winit_event::WindowEvent::PointerLeft {
                device_id,
                position,
                primary,
                kind,
            } => {
                let position = position.map(|pos| {
                    // move by x_origin and y_origin
                    let window_state = window.state.lock().unwrap();
                    let window_state = window_state.as_ref().unwrap();
                    let window_size = window_state.size;
                    let pos = (pos.x as f32, pos.y as f32);

                    (
                        pos.0 - (window_size.width as f32 / 2.0),
                        pos.1 - (window_size.height as f32 / 2.0),
                    )
                });

                match kind {
                    // this is a touch even
                    winit_event::PointerKind::Touch(finger_id) => Event::TouchEnd {
                        timestamp: timestamp.into(),
                        // position should always be Some for touch events
                        position: position.expect("Position should be Some"),
                        window: window.clone(),
                        id: Some(finger_id.into_raw() as u64),
                    },
                    // this is a mouse event
                    _ => Event::CursorExited {
                        timestamp: timestamp.into(),
                        window: window.clone(),
                    },
                }
            }
            winit_event::WindowEvent::PointerMoved {
                device_id,
                position,
                primary,
                source,
            } => {
                //  let position = (Size::Pixels(position.x) - Size::ScreenWidth(0.5), Size::Pixels(-position.y) + Size::ScreenHeight(0.5));
                let position = (position.x as f32, position.y as f32);

                // move by x_origin and y_origin
                let window_state = window.state.lock().unwrap();
                let window_state = window_state.as_ref().unwrap();
                let window_size = window_state.size;
                let position = (
                    position.0 - (window_size.width as f32 / 2.0),
                    position.1 - (window_size.height as f32 / 2.0),
                );

                match source {
                    // this is a touch event
                    winit_event::PointerSource::Touch { finger_id, force } => Event::TouchMove {
                        timestamp: timestamp.into(),
                        position,
                        window: window.clone(),
                        id: Some(finger_id.into_raw() as u64),
                    },
                    // this is a mouse event
                    _ => Event::CursorMoved {
                        timestamp: timestamp.into(),
                        position,
                        window: window.clone(),
                    },
                }
            }
            // match focus events
            winit_event::WindowEvent::Focused(focused) => {
                if focused {
                    Event::FocusGained {
                        timestamp: timestamp.into(),
                        window: window.clone(),
                    }
                } else {
                    Event::FocusLost {
                        timestamp: timestamp.into(),
                        window: window.clone(),
                    }
                }
            }

            // // match cursor enter events
            // winit_event::WindowEvent::PointerEntered { .. } => Event::CursorEntered {
            //     timestamp: timestamp.into(),
            //     window: window.clone(),
            // },
            // // match cursor exit events
            // winit_event::WindowEvent::PointerLeft { .. } => Event::CursorExited {
            //     timestamp: timestamp.into(),
            //     window: window.clone(),
            // },
            // match touchpad press events
            winit_event::WindowEvent::TouchpadPressure {
                device_id: _,
                pressure,
                stage,
            } => Event::TouchpadPress {
                timestamp: timestamp.into(),
                pressure,
                stage,
                window: window.clone(),
            },
            // match any other event
            _ => Event::Other {
                timestamp: timestamp.into(),
                name: format!("{:?}", event),
            },
        };

        Ok(data)
    }
}

/// Receives physical input events.
#[derive(Debug)]
#[pyclass]
pub struct EventReceiver {
    pub(crate) receiver: async_broadcast::Receiver<Event>,
}

/// Contains a vector of events.
#[derive(Debug, Clone)]
#[pyclass]
pub struct EventVec(Vec<Event>);

// convenience methods for KeyEventVec
impl EventVec {
    /// Check if the given EventVec contains the provided key in the
    /// `Pressed` state (convenience method).
    pub fn key_pressed(&self, key: &str) -> bool {
        self.iter().any(|key_event| key_event.key_pressed(key))
    }

    /// Check if the given EventVec contains the provided key in the
    /// `Released` state (convenience method).
    pub fn key_released(&self, key: &str) -> bool {
        self.iter().any(|key_event| key_event.key_released(key))
    }

    /// Check if the given EventVec contains a released left mouse button.
    pub fn left_mouse_button_released(&self) -> bool {
        self.iter().any(Event::left_mouse_button_released)
    }

    /// Check if the given EventVec contains a pressed left mouse button.
    pub fn left_mouse_button_pressed(&self) -> bool {
        self.iter().any(Event::left_mouse_button_pressed)
    }

    /// Check if the given EventVec contains a pressed right mouse button.
    pub fn right_mouse_button_pressed(&self) -> bool {
        self.iter().any(Event::right_mouse_button_pressed)
    }

    /// Check if the given EventVec contains a released right mouse button.
    pub fn right_mouse_button_released(&self) -> bool {
        self.iter().any(Event::right_mouse_button_released)
    }
}

#[pymethods]
impl EventVec {
    /// Check if the given KeyEventVec contains the provided key in the
    /// `Pressed` state (convenience method).
    #[pyo3(name = "key_pressed")]
    pub fn py_key_pressed(&self, key: &str) -> bool {
        self.key_pressed(key)
    }

    /// Check if the given KeyEventVec contains the provided key in the
    /// `Released` state (convenience method).
    #[pyo3(name = "key_released")]
    pub fn py_key_released(&self, key: &str) -> bool {
        self.key_released(key)
    }

    /// Check if the given KeyEventVec contains a released left mouse button.
    #[pyo3(name = "left_mouse_button_released")]
    pub fn py_left_mouse_button_released(&self) -> bool {
        self.left_mouse_button_released()
    }

    /// Check if the given KeyEventVec contains a pressed left mouse button.
    #[pyo3(name = "left_mouse_button_pressed")]
    pub fn py_left_mouse_button_pressed(&self) -> bool {
        self.left_mouse_button_pressed()
    }

    /// Check if the given KeyEventVec contains a pressed right mouse button.
    #[pyo3(name = "right_mouse_button_pressed")]
    pub fn py_right_mouse_button_pressed(&self) -> bool {
        self.right_mouse_button_pressed()
    }

    /// Check if the given KeyEventVec contains a released right mouse button.
    #[pyo3(name = "right_mouse_button_released")]
    pub fn py_right_mouse_button_released(&self) -> bool {
        self.right_mouse_button_released()
    }

    /// Returns all pressed keys in the KeyEventVec.
    #[pyo3(name = "keys_pressed")]
    pub fn py_keys_pressed(&self) -> Vec<String> {
        self.iter()
            .filter_map(|event| match event {
                Event::KeyPress { key, .. } => Some(key.clone()),
                _ => None,
            })
            .collect()
    }

    /// Returns all released keys in the KeyEventVec.
    #[pyo3(name = "keys_released")]
    pub fn py_keys_released(&self) -> Vec<String> {
        self.iter()
            .filter_map(|event| match event {
                Event::KeyRelease { key, .. } => Some(key.clone()),
                _ => None,
            })
            .collect()
    }

    pub fn events(&self) -> Vec<Event> {
        self.0.clone()
    }
}

// make KeyEventVec behave like a vector of KeyEvents
impl Deref for EventVec {
    type Target = Vec<Event>;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl EventReceiver {
    pub fn poll(&mut self) -> EventVec {
        let mut inputs = Vec::new();
        while let Ok(input) = self.receiver.try_recv() {
            inputs.push(input);
        }
        EventVec(inputs)
    }

    /// Flushes the internal buffer of key events for this receiver without
    /// returning them. This is slightly more efficient than calling
    /// `get_keys` and ignoring the result.
    pub fn flush(&mut self) {
        while let Ok(_) = self.receiver.try_recv() {}
    }
}

#[pymethods]
impl EventReceiver {
    /// Polls the receiver for new events.
    #[pyo3(name = "poll")]
    pub fn py_poll(&mut self) -> EventVec {
        self.poll()
    }

    /// Flushes the internal buffer of key events for this receiver without
    /// returning them. This is slightly more efficient than calling
    /// `poll` and ignoring the result.
    #[pyo3(name = "flush")]
    pub fn py_flush(&mut self) {
        self.flush()
    }
}

pub(crate) type EventHandlerId = usize;

pub(crate) type EventHandler = Arc<dyn Fn(Event) -> bool + Send + Sync>;
/// Extension for tvpes
pub trait EventHandlingExt {
    /// Add an event handler to handle a specific event type.
    fn add_event_handler<F>(&self, kind: EventKind, handler: F) -> EventHandlerId
    where
        F: Fn(Event) -> bool + 'static + Send + Sync;

    /// Remove an event handler.
    fn remove_event_handler(&self, id: EventHandlerId);

    /// Dispatch an event. Returns a boolean indicating whether the event was
    /// consumed by any of the handlers.
    fn dispatch_event(&self, event: Event) -> bool;
}

impl FromPyObject<'_> for EventKind {
    fn extract_bound(ob: &pyo3::Bound<'_, pyo3::PyAny>) -> pyo3::PyResult<Self> {
        let kind = ob.extract::<String>()?;
        // try to convert the string to an EventKind
        let kind = EventKind::from_str(&kind).map_err(|_| {
            let possible_kinds = EventKind::VARIANTS
                .iter()
                .map(|v| format!("{:}", v))
                .collect::<Vec<_>>()
                .join(", ");
            pyo3::exceptions::PyValueError::new_err(format!(
                "Invalid event kind '{:}'. Possible values are: {:}",
                kind, possible_kinds
            ))
        })?;

        Ok(kind)
    }
}

impl<'py> IntoPyObject<'py> for EventKind {
    type Target = PyString;
    type Output = Bound<'py, Self::Target>;
    type Error = Infallible;

    #[inline]
    fn into_pyobject(self, py: Python<'py>) -> Result<Self::Output, Self::Error> {
        let kind = self.to_string();
        let kind = PyString::new(py, &kind);
        Ok(kind)
    }
}
