from __future__ import annotations

import functools
from abc import ABC, abstractmethod
from collections.abc import AsyncGenerator, AsyncIterator, Awaitable, Callable
from dataclasses import dataclass, field
from typing import Any, TypeVar

from flexai.llm.exceptions import LLMException, map_exception
from flexai.message import AIMessage, Message, MessageContent, SystemMessage
from flexai.tool import Tool

T = TypeVar("T")


def _handle_provider_exceptions_coroutine(
    func: Callable[..., Awaitable[T]], provider: str
) -> Callable[..., Awaitable[T]]:
    """Handle provider-specific exceptions and convert them to generic LLMExceptions.

    Args:
        func: The coroutine function to wrap
        provider: The provider name ("openai", "anthropic", "gemini", etc.)

    Returns:
        A wrapper around the input function that does the improved exception handling
    """

    @functools.wraps(func)
    async def wrapper(*args: Any, **kwargs: Any) -> T:
        try:
            return await func(*args, **kwargs)
        except Exception as e:
            # Skip our own exceptions
            if isinstance(e, LLMException):
                raise
            # Map provider-specific exceptions to our generic ones
            mapped_exception = map_exception(e, provider)
            raise mapped_exception from None

    return wrapper


def _handle_provider_exceptions_generator(
    func: Callable[..., AsyncIterator[T]], provider: str
) -> Callable[..., AsyncIterator[T]]:
    """Handle provider-specific exceptions and convert them to generic LLMExceptions for async generators.

    Args:
        func: The async generator function to wrap
        provider: The provider name ("openai", "anthropic", "gemini", etc.)

    Returns:
        A wrapper around the input function that does the improved exception handling
    """

    @functools.wraps(func)
    async def wrapper(*args: Any, **kwargs: Any) -> AsyncGenerator[T, None]:
        try:
            async for item in func(*args, **kwargs):
                yield item
        except Exception as e:
            # Skip our own exceptions
            if isinstance(e, LLMException):
                raise
            # Map provider-specific exceptions to our generic ones
            mapped_exception = map_exception(e, provider)
            raise mapped_exception from None

    return wrapper


@dataclass(frozen=True)
class Client(ABC):
    """Abstract base class for language model clients.

    Defines the interface for interacting with various language models.
    Subclasses should implement the necessary methods for specific LLM providers.
    """

    # Provider name (e.g., "openai", "anthropic", "gemini")
    provider: str = field(kw_only=True)

    @abstractmethod
    async def get_chat_response(
        self,
        messages: list[Message],
        *,
        system: str | SystemMessage = "",
        tools: list[Tool] | None = None,
        force_tool: bool = True,
        disable_thinking: bool = False,
    ) -> AIMessage:
        """Retrieve a response from the chat model.

        Args:
            messages: Conversation history to send to the model.
            system: Optional system message to set the behavior of the AI.
            tools: Optional list of tools available for the model to use.
            force_tool: Boolean flag for whether to fully interpret tool calls/results in the history.
            disable_thinking: Boolean flag for whether to disable thinking.

        Returns:
            A list of AI-generated messages in response to the input.

        Raises:
            LLMException: For various generic LLM-related errors
        """
        # This method will be implemented by subclasses and automatically wrapped with exception handling

    @abstractmethod
    def stream_chat_response(
        self,
        messages: list[Message],
        *,
        system: str | SystemMessage = "",
        tools: list[Tool] | None = None,
        allow_tool: bool = True,
        force_tool: bool = True,
        disable_thinking: bool = False,
    ) -> AsyncIterator[MessageContent | AIMessage]:
        """Stream the response from the chat model in real-time.

        Args:
            messages: Conversation history to send to the model.
            system: Optional system message to set the behavior of the AI.
            tools: Optional list of tools available for the model to use.
            allow_tool: Boolean flag for whether to fully interpret tool calls/results in the history.
            force_tool: Boolean flag for whether to fully interpret tool calls/results in the history.
            disable_thinking: Boolean flag for whether to disable thinking.

        Yields:
            AI-generated messages as they are generated by the model.

        Raises:
            LLMException: For various generic LLM-related errors
        """
        # This method will be implemented by subclasses and automatically wrapped with exception handling

    def __init_subclass__(cls, **kwargs):
        super().__init_subclass__(**kwargs)

        wrapped = _handle_provider_exceptions_coroutine(
            cls.get_chat_response, cls.provider
        )
        cls.get_chat_response = wrapped  # pyright: ignore[reportAttributeAccessIssue]

        wrapped = _handle_provider_exceptions_generator(
            cls.stream_chat_response, cls.provider
        )
        cls.stream_chat_response = wrapped
