'''Everything in this file is generated by an AI, not me'''

import math
from typing import Optional

class GEMINI_SequencePredictor:
    """
    Analyzes the specific sequence [1.25, 3, 4, 5, 7, 10, ..., 20].
    
    The class generates the full finite sequence and provides a method to find 
    the next value in the sequence based on an arbitrary input 'x', including 
    extrapolation if 'x' exceeds the defined sequence limit (20.0).
    """
    def __init__(self):
        """
        Generates the complete sequence, which is stored internally.
        The pattern: differences increase by 1 (+1, +2, +3, ...) starting 
        from the 4th term (5), and the sequence is capped at 20.
        """
        # 1. Initialize with the known starting numbers
        sequence: list[float] = [1.25, 3.0, 4.0]
        current_value = 4.0  
        difference = 1       # Initial difference for the pattern (4 + 1 = 5)
        
        # 2. Loop to apply the increasing difference pattern up to the limit 20
        while True:
            next_value = current_value + difference

            # Check the stopping condition: 20 must be the final number.
            if next_value >= 20.0:
                # If 20 is not already the last element, add it and stop.
                if sequence[-1] != 20.0:
                    sequence.append(20.0)
                break
            else:
                # Add the number and increment the difference for the next step.
                sequence.append(next_value)
                current_value = next_value
                difference += 1
                
        # The final sequence: [1.25, 3.0, 4.0, 5.0, 7.0, 10.0, 14.0, 19.0, 20.0]
        self.sequence = sequence
        # Store the state for extrapolation: The last natural pattern step was 19.0, 
        # and the next difference *should* be 6.
        self.last_natural_term = 19.0
        self.next_difference_start = 6.0
        
        print(f"Generated Finite Sequence: {self.sequence}")

    def _extrapolate_next_value(self, input_x: float) -> float:
        """
        Handles the case where input_x > 20.0 by continuing the sequence pattern
        (difference +1) until the predicted term is greater than input_x.
        """
        current_val = self.last_natural_term
        difference = self.next_difference_start
        
        while True:
            next_val = current_val + difference
            
            if next_val > input_x:
                # This is the next theoretical sequence number that follows input_x
                print(f"(Extrapolated Anchor: {current_val} with Diff {difference}, Prediction: {next_val})")
                return next_val
            
            current_val = next_val
            difference += 1.0

    def get_next_sequence_value(self, input_x: float) -> float:
        """
        Finds the closest match 'c' to 'input_x' in the sequence and returns 
        the next value. If 'input_x' is greater than 20.0, it extrapolates the 
        sequence pattern to find the prediction.

        Args:
            input_x (float): The user's arbitrary input number.

        Returns:
            float: The next value in the sequence after the anchor point.
        """
        # --- CASE 1: INPUT EXCEEDS THE FINITE SEQUENCE LIMIT (20.0) ---
        if input_x > 20.0:
            print(f"Input {input_x} exceeds the sequence limit (20.0). Entering extrapolation mode.")
            return self._extrapolate_next_value(input_x)

        # --- CASE 2: INPUT IS WITHIN THE FINITE SEQUENCE (<= 20.0) ---
        
        anchor_index = 0
        min_difference = abs(input_x - self.sequence[0])
        
        # 1. Find the Anchor Point (closest match in the sequence)
        for i, number in enumerate(self.sequence):
            current_difference = abs(input_x - number)
            
            # Tie-breaker logic: Use "<=" to favor the larger number.
            if current_difference < min_difference:
                min_difference = current_difference
                anchor_index = i
            elif current_difference == min_difference:
                # Explicitly ensure the larger value is chosen in case of a tie
                if number > self.sequence[anchor_index]:
                     anchor_index = i

        # 2. Determine the Next Value
        
        anchor_value = self.sequence[anchor_index]
        print(f"Input {input_x} anchors at {anchor_value} (Index {anchor_index}).")
        
        # If the anchor is the last element (20.0), return 20.0 as the "next" 
        # since the finite list ends here.
        if anchor_index == len(self.sequence) - 1:
            print("Anchor is the final term. Returning the final term.")
            return anchor_value
        else:
            # Otherwise, return the element at the next index
            next_value = self.sequence[anchor_index + 1]
            return next_value

# --- Example Usage ---
'''
if __name__ == "__main__":
    predictor = SequencePredictor()
    print("\n--- Next Value Predictions ---")

    # Example 1: Standard case (Input 7.0)
    input1 = 7.0
    next1 = predictor.get_next_sequence_value(input1)
    print(f"Next value after {input1} is: {next1}\n") # Expected: 10.0 (Anchors at 7.0)

    # Example 2: Extrapolation Case 1 (Input 21.0 > 20.0)
    # Theoretical sequence continues: 19 + 6 = 25. Since 25 > 21, the prediction is 25.
    input2 = 21.0
    next2 = predictor.get_next_sequence_value(input2)
    print(f"Next value after {input2} is: {next2}\n") # Expected: 25.0

    # Example 3: Extrapolation Case 2 (Input 26.0)
    # Theoretical sequence continues: 19 + 6 = 25. 25 < 26. Next step: 25 + 7 = 32.
    input3 = 26.0
    next3 = predictor.get_next_sequence_value(input3)
    print(f"Next value after {input3} is: {next3}\n") # Expected: 32.0

    # Example 4: Edge Case (Input 19.5 - Closest to 20.0)
    input4 = 19.5
    next4 = predictor.get_next_sequence_value(input4)
    print(f"Next value after {input4} is: {next4}\n") # Expected: 20.0 (Anchors at 20.0)
'''
