import os
import subprocess
import sys
from typing import List, Optional
from urllib.parse import urlparse

import click

from launchable.utils.launchable_client import LaunchableClient
from launchable.utils.tracking import Tracking, TrackingClient

from ...app import Application
from ...utils.commands import Command
from ...utils.commit_ingester import upload_commits
from ...utils.env_keys import COMMIT_TIMEOUT, REPORT_ERROR_KEY
from ...utils.fail_fast_mode import set_fail_fast_mode, warn_and_exit_if_fail_fast_mode
from ...utils.git_log_parser import parse_git_log
from ...utils.http_client import get_base_url
from ...utils.java import cygpath, get_java_command
from ...utils.logger import LOG_LEVEL_AUDIT, Logger

jar_file_path = os.path.normpath(os.path.join(os.path.dirname(__file__), "../../jar/exe_deploy.jar"))


@click.command()
@click.option(
    '--source',
    help="repository path",
    default=os.getcwd(),
    type=click.Path(exists=True, file_okay=False),
)
@click.option(
    '--executable',
    help="[Obsolete] it was to specify how to perform commit collection but has been removed",
    type=click.Choice(['jar', 'docker']),
    default='jar',
    hidden=True)
@click.option(
    '--max-days',
    help="the maximum number of days to collect commits retroactively",
    default=30)
@click.option(
    '--scrub-pii',
    is_flag=True,
    help='[Deprecated] Scrub emails and names',
    hidden=True)
@click.option(
    '--import-git-log-output',
    help="import from the git-log output",
    type=click.Path(exists=True, dir_okay=False,
                    resolve_path=True, allow_dash=True),
)
@click.pass_context
def commit(ctx, source: str, executable: bool, max_days: int, scrub_pii: bool, import_git_log_output: str):
    if executable == 'docker':
        sys.exit("--executable docker is no longer supported")

    tracking_client = TrackingClient(Command.COMMIT, app=ctx.obj)
    client = LaunchableClient(tracking_client=tracking_client, app=ctx.obj)
    set_fail_fast_mode(client.is_fail_fast_mode())

    if import_git_log_output:
        _import_git_log(import_git_log_output, ctx.obj)
        return

    # Commit messages are not collected in the default.
    is_collect_message = False
    is_collect_files = False
    try:
        res = client.request("get", "commits/collect/options")
        res.raise_for_status()
        is_collect_message = res.json().get("commitMessage", False)
        is_collect_files = res.json().get("files", False)
    except Exception as e:
        tracking_client.send_error_event(
            event_name=Tracking.ErrorEvent.INTERNAL_CLI_ERROR,
            stack_trace=str(e),
            api="commits/options",
        )
        client.print_exception_and_recover(e)

    cwd = os.path.abspath(source)
    try:
        exec_jar(cwd, max_days, ctx.obj, is_collect_message, is_collect_files)
    except Exception as e:
        if os.getenv(REPORT_ERROR_KEY):
            raise e
        else:
            warn_and_exit_if_fail_fast_mode(
                "Couldn't get commit history from `{}`. Do you run command root of git-controlled directory? "
                "If not, please set a directory use by --source option.\nerror: {}".format(cwd, e))


def exec_jar(source: str, max_days: int, app: Application, is_collect_message: bool, is_collect_files: bool):
    java = get_java_command()

    if not java:
        sys.exit("You need to install Java")

    base_url = get_base_url()

    # using subprocess.check_out with shell=False and a list of command to prevent vulnerability
    # https://knowledge-base.secureflag.com/vulnerabilities/code_injection/os_command_injection_python.html
    command = [java]
    command.extend(_build_proxy_option(os.getenv("HTTPS_PROXY")))
    command.extend([
        "-jar",
        cygpath(jar_file_path),
        "ingest:commit",
        "-endpoint",
        "{}/intake/".format(base_url),
        "-max-days",
        str(max_days)
    ])

    if Logger().logger.isEnabledFor(LOG_LEVEL_AUDIT):
        command.append("-audit")
    if app.dry_run:
        command.append("-dry-run")
    if app.skip_cert_verification:
        command.append("-skip-cert-verification")
    if is_collect_message:
        command.append("-commit-message")
    if is_collect_files:
        command.append("-files")
    if os.getenv(COMMIT_TIMEOUT):
        command.append("-enable-timeout")
    command.append(cygpath(source))

    subprocess.run(
        command,
        check=True,
        shell=False,
    )


def _import_git_log(output_file: str, app: Application):
    try:
        with click.open_file(output_file) as fp:
            commits = parse_git_log(fp)
        upload_commits(commits, app)
    except Exception as e:
        if os.getenv(REPORT_ERROR_KEY):
            raise e
        else:
            warn_and_exit_if_fail_fast_mode("Failed to import the git-log output\n error: {}".format(e))


def _build_proxy_option(https_proxy: Optional[str]) -> List[str]:
    if not https_proxy:
        return []

    if not (https_proxy.startswith("https://") or https_proxy.startswith("http://")):
        https_proxy = "https://" + https_proxy
    proxy_url = urlparse(https_proxy)

    options = []
    if proxy_url.hostname:
        options.append("-Dhttps.proxyHost={}".format(proxy_url.hostname))
    if proxy_url.port:
        options.append("-Dhttps.proxyPort={}".format(proxy_url.port))
    return options
