import glob
import os
from typing import Dict, List, Optional

import click

from launchable.utils import glob as uglob
from launchable.utils.java import junit5_nested_class_path_builder

from . import launchable

# Surefire has the default inclusion pattern
# https://maven.apache.org/surefire/maven-surefire-plugin/test-mojo.html#includes
# and the default exclusion pattern
# https://maven.apache.org/surefire/maven-surefire-plugin/test-mojo.html#excludes
# these variables emulates those effects.
# TODO: inclusion/exclusion are user configurable patterns, so it should be user configurable
# beyond that and to fully generalize this, there's internal discussion of
# this at https://launchableinc.atlassian.net/l/c/TXDJnn09
includes = [uglob.compile(x) for x in [
    # HACK: we check extensions outside the glob. We seem to allow both source
    # file enumeration and class file enumeration
    '**/Test*.*',
    '**/*Test.*',
    '**/*Spec.*',
    '**/*Tests.*',
    '**/*TestCase.*'
]]
excludes = [uglob.compile(x) for x in [
    '**/*$*'
]]

# Test if a given path name is a test that Surefire recognizes


def is_file(f: str) -> bool:
    if not (f.endswith('.java') or f.endswith(".scala") or f.endswith(".kt") or f.endswith(".class") or f.endswith(".groovy")):
        return False
    for p in excludes:
        if p.fullmatch(f):
            return False
    for p in includes:
        if p.fullmatch(f):
            return True
    return False


@click.option(
    '--test-compile-created-file',
    'test_compile_created_file',
    required=False,
    multiple=True,
    type=click.Path(exists=True),
    help="Please run `mvn test-compile` command to create input file for this option",
)
@click.option(
    '--scan-test-compile-lst',
    'is_scan_test_compile_lst',
    required=False,
    is_flag=True,
    help="Scan testCompile/default-testCompile/createdFiles.lst for *.lst files generated by `mvn compile` and use them as test inputs.",  # noqa: E501
)
@click.argument('source_roots', required=False, nargs=-1)
@launchable.subset
def subset(client, source_roots, test_compile_created_file, is_scan_test_compile_lst):

    def file2class_test_path(f: str) -> List[Dict[str, str]]:
        # remove extension
        f, _ = os.path.splitext(f)

        # directory -> package name conversion
        cls_name = f.replace(os.path.sep, '.')
        return [{"type": "class", "name": cls_name}]

    def file2test(f: str) -> Optional[List]:
        if is_file(f):
            return file2class_test_path(f)
        else:
            return None

    files_to_read = list(test_compile_created_file)
    if is_scan_test_compile_lst:
        if len(test_compile_created_file) > 0:
            click.echo(click.style(
                "Warning: --test-compile-created-file is overridden by --scan-test-compile-lst", fg="yellow"),
                err=True)

        pattern = os.path.join('**', 'createdFiles.lst')
        files_to_read = glob.glob(pattern, recursive=True)

        if not files_to_read:
            click.echo(click.style(
                "Warning: No .lst files. Please run after executing `mvn test-compile`", fg="yellow"),
                err=True)
            return

    if files_to_read:
        for file in files_to_read:
            with open(file, 'r') as f:
                lines = f.readlines()
                if len(lines) == 0:
                    click.echo(click.style(
                        "Warning: --test-compile-created-file {} is empty".format(file), fg="yellow"),
                        err=True)

                for l in lines:
                    # trim trailing newline
                    l = l.strip()

                    path = file2test(l)
                    if path:
                        client.test_paths.append(path)
    else:
        for root in source_roots:
            client.scan(root, '**/*', file2test)

    client.run()


@launchable.split_subset
def split_subset(client):
    def format_same_bin(s: str) -> List[Dict[str, str]]:
        return [{"type": "class", "name": s}]

    client.same_bin_formatter = format_same_bin
    client.run()


# TestNG produces surefire-reports/testng-results.xml in TestNG's native format.
# Surefire produces TEST-*.xml in JUnit format (see Surefire's StatelessXmlReporter.getReportFile)
# In addition, TestNG also produces surefire-reports/junitreports/TEST-*.xml
# (see TestNG's JUnitReportReporter.getFileName)
# And there are more test reports in play.
#
# So to collectly find tests without duplications, we need to find surefire-reports/TEST-*.xml
# not surefire-reports/**/TEST-*.xml nor surefire-reports/*.xml
@click.argument('reports', required=True, nargs=-1)
@launchable.record.tests
def record_tests(client, reports):
    client.path_builder = junit5_nested_class_path_builder(client.path_builder)
    launchable.CommonRecordTestImpls.load_report_files(client=client, source_roots=reports)
