"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PropagatedTagSource = exports.DeploymentControllerType = exports.LaunchType = exports.BaseService = exports.ListenerConfig = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const core_1 = require("@aws-cdk/core");
const task_definition_1 = require("../base/task-definition");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const props = this.props || {};
        const protocol = props.protocol;
        const port = props.port !== undefined ? props.port : (protocol === undefined ? 80 :
            (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80));
        this.listener.addTargets(id, {
            ...props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const port = this.props !== undefined ? this.props.port : 80;
        this.listener.addTargets(id, {
            ...this.props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        this.taskDefinition = taskDefinition;
        this.resource = new ecs_generated_1.CfnService(this, 'Service', {
            desiredCount: props.desiredCount,
            serviceName: this.physicalName,
            loadBalancers: core_1.Lazy.anyValue({ produce: () => this.loadBalancers }, { omitEmptyArray: true }),
            deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent,
            },
            propagateTags: props.propagateTags === PropagatedTagSource.NONE ? undefined : props.propagateTags,
            enableEcsManagedTags: props.enableECSManagedTags === undefined ? false : props.enableECSManagedTags,
            deploymentController: props.deploymentController,
            launchType: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === DeploymentControllerType.EXTERNAL ? undefined : props.launchType,
            healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod),
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.anyValue({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }),
            serviceRegistries: core_1.Lazy.anyValue({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (((_b = props.deploymentController) === null || _b === void 0 ? void 0 : _b.type) === DeploymentControllerType.EXTERNAL) {
            this.node.addWarning('taskDefinition and launchType are blanked out when using external deployment controller.');
        }
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets('ECS', {
     *   port: 80,
     *   targets: [service.loadBalancerTarget({
     *     containerName: 'MyContainer',
     *     containerPort: 1234,
     *   })],
     * });
     */
    loadBalancerTarget(options) {
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            },
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerName: 'web',
     *     containerPort: 80,
     *     newTargetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol,
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', {
            serviceNamespace: appscaling.ServiceNamespace.ECS,
            resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`,
            dimension: 'ecs:service:DesiredCount',
            role: this.makeAutoScalingRole(),
            ...props,
        });
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        const sdNamespace = options.cloudMapNamespace !== undefined ? options.cloudMapNamespace : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error('Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.');
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error('SRV records must be used when network mode is Bridge or Host.');
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        // If the task definition that your service task specifies uses the AWSVPC network mode and a type SRV DNS record is
        // used, you must specify a containerName and containerPort combination
        const containerName = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerName : undefined;
        const containerPort = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerPort : undefined;
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 },
            dnsTtl: options.dnsTtl,
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort,
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName },
            ...props,
        }).attachTo(this);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     * @deprecated use configureAwsVpcNetworkingWithSecurityGroups instead.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.listValue({ produce: () => [securityGroup.securityGroupId] }),
            },
        };
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworkingWithSecurityGroups(vpc, assignPublicIp, vpcSubnets, securityGroups) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroups === undefined || securityGroups.length === 0) {
            securityGroups = [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc })];
        }
        securityGroups.forEach((sg) => { this.connections.addSecurityGroup(sg); }, this);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: securityGroups.map((sg) => sg.securityGroupId),
            },
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.');
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.');
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort,
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName,
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.anyValue({
            produce: () => providedHealthCheckGracePeriod !== undefined ? providedHealthCheckGracePeriod.toSeconds() :
                this.loadBalancers.length > 0 ? 60 :
                    undefined,
        });
    }
}
exports.BaseService = BaseService;
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * The deployment controller type to use for the service.
 */
var DeploymentControllerType;
(function (DeploymentControllerType) {
    /**
     * The rolling update (ECS) deployment type involves replacing the current
     * running version of the container with the latest version.
     */
    DeploymentControllerType["ECS"] = "ECS";
    /**
     * The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy
     */
    DeploymentControllerType["CODE_DEPLOY"] = "CODE_DEPLOY";
    /**
     * The external (EXTERNAL) deployment type enables you to use any third-party deployment controller
     */
    DeploymentControllerType["EXTERNAL"] = "EXTERNAL";
})(DeploymentControllerType = exports.DeploymentControllerType || (exports.DeploymentControllerType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
//# sourceMappingURL=data:application/json;base64,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