"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.volumes }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({ produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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