import * as autoscaling from '@aws-cdk/aws-autoscaling';
import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as cloudmap from '@aws-cdk/aws-servicediscovery';
import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
/**
 * The properties used to define an ECS cluster.
 */
export interface ClusterProps {
    /**
     * The name for the cluster.
     *
     * @default CloudFormation-generated name
     */
    readonly clusterName?: string;
    /**
     * The VPC where your ECS instances will be running or your ENIs will be deployed
     *
     * @default - creates a new VPC with two AZs
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The service discovery namespace created in this cluster
     *
     * @default - no service discovery namespace created, you can use `addDefaultCloudMapNamespace` to add a
     * default service discovery namespace later.
     */
    readonly defaultCloudMapNamespace?: CloudMapNamespaceOptions;
    /**
     * The ec2 capacity to add to the cluster
     *
     * @default - no EC2 capacity will be added, you can use `addCapacity` to add capacity later.
     */
    readonly capacity?: AddCapacityOptions;
    /**
     * If true CloudWatch Container Insights will be enabled for the cluster
     *
     * @default - Container Insights will be disabled for this cluser.
     */
    readonly containerInsights?: boolean;
}
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
export declare class Cluster extends Resource implements ICluster {
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * Manage the allowed network connections for the cluster with Security Groups.
     */
    readonly connections: ec2.Connections;
    /**
     * The VPC associated with the cluster.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     */
    readonly clusterArn: string;
    /**
     * The name of the cluster.
     */
    readonly clusterName: string;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     */
    private _defaultCloudMapNamespace?;
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     */
    private _hasEc2Capacity;
    /**
     * The autoscaling group for added Ec2 capacity
     */
    private _autoscalingGroup?;
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope: Construct, id: string, props?: ClusterProps);
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options: CloudMapNamespaceOptions): cloudmap.INamespace;
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace(): cloudmap.INamespace | undefined;
    /**
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id: string, options: AddCapacityOptions): autoscaling.AutoScalingGroup;
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup: autoscaling.AutoScalingGroup, options?: AddAutoScalingGroupCapacityOptions): void;
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup(): autoscaling.IAutoScalingGroup | undefined;
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity(): boolean;
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * ECS-optimized Windows version list
 */
export declare enum WindowsOptimizedVersion {
    SERVER_2019 = "2019",
    SERVER_2016 = "2016"
}
/**
 * The properties that define which ECS-optimized AMI is used.
 *
 * @deprecated see {@link EcsOptimizedImage}
 */
export interface EcsOptimizedAmiProps {
    /**
     * The Amazon Linux generation to use.
     *
     * @default AmazonLinuxGeneration.AmazonLinux2
     */
    readonly generation?: ec2.AmazonLinuxGeneration;
    /**
     * The Windows Server version to use.
     *
     * @default none, uses Linux generation
     */
    readonly windowsVersion?: WindowsOptimizedVersion;
    /**
     * The ECS-optimized AMI variant to use.
     *
     * @default AmiHardwareType.Standard
     */
    readonly hardwareType?: AmiHardwareType;
}
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
export declare class EcsOptimizedAmi implements ec2.IMachineImage {
    private readonly generation?;
    private readonly windowsVersion?;
    private readonly hwType;
    private readonly amiParameterName;
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props?: EcsOptimizedAmiProps);
    /**
     * Return the correct image
     */
    getImage(scope: Construct): ec2.MachineImageConfig;
}
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 */
export declare class EcsOptimizedImage implements ec2.IMachineImage {
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM
     *
     * @param hardwareType ECS-optimized AMI variant to use
     */
    static amazonLinux2(hardwareType?: AmiHardwareType): EcsOptimizedImage;
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM
     */
    static amazonLinux(): EcsOptimizedImage;
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM
     *
     * @param windowsVersion Windows Version to use
     */
    static windows(windowsVersion: WindowsOptimizedVersion): EcsOptimizedImage;
    private readonly generation?;
    private readonly windowsVersion?;
    private readonly hwType?;
    private readonly amiParameterName;
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    private constructor();
    /**
     * Return the correct image
     */
    getImage(scope: Construct): ec2.MachineImageConfig;
}
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
export interface ICluster extends IResource {
    /**
     * The name of the cluster.
     * @attribute
     */
    readonly clusterName: string;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     * @attribute
     */
    readonly clusterArn: string;
    /**
     * The VPC associated with the cluster.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Manage the allowed network connections for the cluster with Security Groups.
     */
    readonly connections: ec2.Connections;
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     */
    readonly hasEc2Capacity: boolean;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     */
    readonly defaultCloudMapNamespace?: cloudmap.INamespace;
    /**
     * The autoscaling group added to the cluster if capacity is associated to the cluster
     */
    readonly autoscalingGroup?: autoscaling.IAutoScalingGroup;
}
/**
 * The properties to import from the ECS cluster.
 */
export interface ClusterAttributes {
    /**
     * The name of the cluster.
     */
    readonly clusterName: string;
    /**
     * The Amazon Resource Name (ARN) that identifies the cluster.
     *
     * @default Derived from clusterName
     */
    readonly clusterArn?: string;
    /**
     * The VPC associated with the cluster.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The security groups associated with the container instances registered to the cluster.
     */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * Specifies whether the cluster has EC2 instance capacity.
     *
     * @default true
     */
    readonly hasEc2Capacity?: boolean;
    /**
     * The AWS Cloud Map namespace to associate with the cluster.
     *
     * @default - No default namespace
     */
    readonly defaultCloudMapNamespace?: cloudmap.INamespace;
    /**
     * Autoscaling group added to the cluster if capacity is added
     *
     * @default - No default autoscaling group
     */
    readonly autoscalingGroup?: autoscaling.IAutoScalingGroup;
}
/**
 * The properties for adding an AutoScalingGroup.
 */
export interface AddAutoScalingGroupCapacityOptions {
    /**
     * Specifies whether the containers can access the container instance role.
     *
     * @default false
     */
    readonly canContainersAccessInstanceRole?: boolean;
    /**
     * The time period to wait before force terminating an instance that is draining.
     *
     * This creates a Lambda function that is used by a lifecycle hook for the
     * AutoScalingGroup that will delay instance termination until all ECS tasks
     * have drained from the instance. Set to 0 to disable task draining.
     *
     * Set to 0 to disable task draining.
     *
     * @default Duration.minutes(5)
     */
    readonly taskDrainTime?: Duration;
    /**
     * Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.
     * For more information, see [Using Spot Instances](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html).
     *
     * @default false
     */
    readonly spotInstanceDraining?: boolean;
}
/**
 * The properties for adding instance capacity to an AutoScalingGroup.
 */
export interface AddCapacityOptions extends AddAutoScalingGroupCapacityOptions, autoscaling.CommonAutoScalingGroupProps {
    /**
     * The EC2 instance type to use when launching instances into the AutoScalingGroup.
     */
    readonly instanceType: ec2.InstanceType;
    /**
     * The ECS-optimized AMI variant to use. For more information, see
     * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
     *
     * @default - Amazon Linux 2
     */
    readonly machineImage?: ec2.IMachineImage;
}
/**
 * The options for creating an AWS Cloud Map namespace.
 */
export interface CloudMapNamespaceOptions {
    /**
     * The name of the namespace, such as example.com.
     */
    readonly name: string;
    /**
     * The type of CloudMap Namespace to create.
     *
     * @default PrivateDns
     */
    readonly type?: cloudmap.NamespaceType;
    /**
     * The VPC to associate the namespace with. This property is required for private DNS namespaces.
     *
     * @default VPC of the cluster for Private DNS Namespace, otherwise none
     */
    readonly vpc?: ec2.IVpc;
}
/**
 * The ECS-optimized AMI variant to use. For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 */
export declare enum AmiHardwareType {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     */
    STANDARD = "Standard",
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     */
    GPU = "GPU",
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     */
    ARM = "ARM64"
}
