"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.ContainerDependencyCondition = exports.UlimitName = exports.ContainerDefinition = exports.Secret = void 0;
const cdk = require("@aws-cdk/core");
const task_definition_1 = require("./base/task-definition");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManager(secret, field) {
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends cdk.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The configured container links
         */
        this.links = new Array();
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential !== undefined ? props.essential : true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = this.node.id;
        this.imageConfig = props.image.bind(this, this);
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        props.taskDefinition._linkContainer(this);
        if (props.secrets) {
            this.secrets = [];
            for (const [name, secret] of Object.entries(props.secrets)) {
                if (this.taskDefinition.isFargateCompatible && secret.hasField) {
                    throw new Error(`Cannot specify secret JSON field for a task using the FARGATE launch type: '${name}' in container '${this.node.id}'`);
                }
                secret.grantRead(this.taskDefinition.obtainExecutionRole());
                this.secrets.push({
                    name,
                    valueFrom: secret.arn,
                });
            }
        }
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = {
                        ...pm,
                        hostPort: 0,
                    };
                }
            }
            return pm;
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     */
    findPortMapping(containerPort, protocol) {
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change)
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.anyValue({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.anyValue({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.anyValue({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.anyValue({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.anyValue({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.props.environment && renderKV(this.props.environment, 'name', 'value'),
            secrets: this.secrets,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.listValue({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (this.props.gpuCount !== undefined) ? renderResourceRequirements(this.props.gpuCount) : undefined,
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = new Array();
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval != null ? hc.interval.toSeconds() : 30,
        retries: hc.retries !== undefined ? hc.retries : 3,
        startPeriod: hc.startPeriod && hc.startPeriod.toSeconds(),
        timeout: hc.timeout !== undefined ? hc.timeout.toSeconds() : 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount) {
    if (gpuCount === 0) {
        return undefined;
    }
    return [{
            type: 'GPU',
            value: gpuCount.toString(),
        }];
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
//# sourceMappingURL=data:application/json;base64,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