"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImage = void 0;
/**
 * Constructs for types of container images
 */
class ContainerImage {
    /**
     * Reference an image on DockerHub or another online registry
     */
    static fromRegistry(name, props = {}) {
        return new repository_1.RepositoryImage(name, props);
    }
    /**
     * Reference an image in an ECR repository
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new ecr_1.EcrImage(repository, tag);
    }
    /**
     * Reference an image that's constructed directly from sources on disk.
     *
     * If you already have a `DockerImageAsset` instance, you can use the
     * `ContainerImage.fromDockerImageAsset` method instead.
     *
     * @param directory The directory containing the Dockerfile
     */
    static fromAsset(directory, props = {}) {
        return new asset_image_1.AssetImage(directory, props);
    }
    /**
     * Use an existing `DockerImageAsset` for this container image.
     *
     * @param asset The `DockerImageAsset` to use for this container definition.
     */
    static fromDockerImageAsset(asset) {
        return {
            bind(_scope, containerDefinition) {
                asset.repository.grantPull(containerDefinition.taskDefinition.obtainExecutionRole());
                return {
                    imageName: asset.imageUri,
                };
            },
        };
    }
}
exports.ContainerImage = ContainerImage;
const asset_image_1 = require("./images/asset-image");
const ecr_1 = require("./images/ecr");
const repository_1 = require("./images/repository");
//# sourceMappingURL=data:application/json;base64,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