"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstanceDrainHook = void 0;
const autoscaling = require("@aws-cdk/aws-autoscaling");
const hooks = require("@aws-cdk/aws-autoscaling-hooktargets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
/**
 * A hook to drain instances from ECS traffic before they're terminated
 */
class InstanceDrainHook extends cdk.Construct {
    /**
     * Constructs a new instance of the InstanceDrainHook class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const drainTime = props.drainTime || cdk.Duration.minutes(5);
        // Invoke Lambda via SNS Topic
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline(fs.readFileSync(path.join(__dirname, 'lambda-source', 'index.py'), { encoding: 'utf-8' })),
            handler: 'index.lambda_handler',
            runtime: lambda.Runtime.PYTHON_3_6,
            // Timeout: some extra margin for additional API calls made by the Lambda,
            // up to a maximum of 15 minutes.
            timeout: cdk.Duration.seconds(Math.min(drainTime.toSeconds() + 10, 900)),
            environment: {
                CLUSTER: props.cluster.clusterName,
            },
        });
        // Hook everything up: ASG -> Topic, Topic -> Lambda
        props.autoScalingGroup.addLifecycleHook('DrainHook', {
            lifecycleTransition: autoscaling.LifecycleTransition.INSTANCE_TERMINATING,
            defaultResult: autoscaling.DefaultResult.CONTINUE,
            notificationTarget: new hooks.FunctionHook(fn),
            heartbeatTimeout: drainTime,
        });
        // Describe actions cannot be restricted and restrict the CompleteLifecycleAction to the ASG arn
        // https://docs.aws.amazon.com/autoscaling/ec2/userguide/control-access-using-iam.html
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceStatus',
                'ec2:DescribeHosts',
            ],
            resources: ['*'],
        }));
        // Restrict to the ASG
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['autoscaling:CompleteLifecycleAction'],
            resources: [props.autoScalingGroup.autoScalingGroupArn],
        }));
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['ecs:DescribeContainerInstances', 'ecs:DescribeTasks'],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
        }));
        // Restrict to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:ListContainerInstances',
                'ecs:SubmitContainerStateChange',
                'ecs:SubmitTaskStateChange',
            ],
            resources: [props.cluster.clusterArn],
        }));
        // Restrict the container-instance operations to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:UpdateContainerInstancesState',
                'ecs:ListTasks',
            ],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
            resources: ['*'],
        }));
    }
}
exports.InstanceDrainHook = InstanceDrainHook;
//# sourceMappingURL=data:application/json;base64,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