"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuiltInAttributes = exports.Ec2Service = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
const task_definition_1 = require("../base/task-definition");
/**
 * This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class Ec2Service extends base_service_1.BaseService {
    /**
     * Constructs a new instance of the Ec2Service class.
     */
    constructor(scope, id, props) {
        var _a;
        if (props.daemon && props.desiredCount !== undefined) {
            throw new Error('Daemon mode launches one task on every instance. Don\'t supply desiredCount.');
        }
        if (props.daemon && props.maxHealthyPercent !== undefined && props.maxHealthyPercent !== 100) {
            throw new Error('Maximum percent must be 100 for daemon mode.');
        }
        if (props.minHealthyPercent !== undefined && props.maxHealthyPercent !== undefined && props.minHealthyPercent >= props.maxHealthyPercent) {
            throw new Error('Minimum healthy percent must be less than maximum healthy percent.');
        }
        if (!props.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with EC2');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            // If daemon, desiredCount must be undefined and that's what we want. Otherwise, default to 1.
            desiredCount: props.daemon || props.desiredCount !== undefined ? props.desiredCount : 1,
            maxHealthyPercent: props.daemon && props.maxHealthyPercent === undefined ? 100 : props.maxHealthyPercent,
            minHealthyPercent: props.daemon && props.minHealthyPercent === undefined ? 0 : props.minHealthyPercent,
            launchType: base_service_1.LaunchType.EC2,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            placementConstraints: core_1.Lazy.anyValue({ produce: () => this.constraints }, { omitEmptyArray: true }),
            placementStrategies: core_1.Lazy.anyValue({ produce: () => this.strategies }, { omitEmptyArray: true }),
            schedulingStrategy: props.daemon ? 'DAEMON' : 'REPLICA',
        }, props.taskDefinition);
        this.constraints = [];
        this.strategies = [];
        this.daemon = props.daemon || false;
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        if (props.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroups from ASG.
            // We have to be smart here -- by default future Security Group rules would be created
            // in the Cluster stack. However, if the Cluster is in a different stack than us,
            // that will lead to a cyclic reference (we point to that stack for the cluster name,
            // but that stack will point to the ALB probably created right next to us).
            //
            // In that case, reference the same security groups but make sure new rules are
            // created in the current scope (i.e., this stack)
            validateNoNetworkingProps(props);
            this.connections.addSecurityGroup(...securityGroupsInThisStack(this, props.cluster.connections.securityGroups));
        }
        this.addPlacementConstraints(...props.placementConstraints || []);
        this.addPlacementStrategies(...props.placementStrategies || []);
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
    /**
     * Imports from the specified service ARN.
     */
    static fromEc2ServiceArn(scope, id, ec2ServiceArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = ec2ServiceArn;
                this.serviceName = core_1.Stack.of(scope).parseArn(ec2ServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports from the specified service attrributes.
     */
    static fromEc2ServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     */
    addPlacementStrategies(...strategies) {
        if (strategies.length > 0 && this.daemon) {
            throw new Error("Can't configure placement strategies when daemon=true");
        }
        for (const strategy of strategies) {
            this.strategies.push(...strategy.toJson());
        }
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     */
    addPlacementConstraints(...constraints) {
        for (const constraint of constraints) {
            this.constraints.push(...constraint.toJson());
        }
    }
    /**
     * Validates this Ec2Service.
     */
    validate() {
        const ret = super.validate();
        if (!this.cluster.hasEc2Capacity) {
            ret.push('Cluster for this service needs Ec2 capacity. Call addXxxCapacity() on the cluster.');
        }
        return ret;
    }
}
exports.Ec2Service = Ec2Service;
/**
 * Validate combinations of networking arguments.
 */
function validateNoNetworkingProps(props) {
    if (props.vpcSubnets !== undefined
        || props.securityGroup !== undefined
        || props.securityGroups !== undefined
        || props.assignPublicIp) {
        throw new Error('vpcSubnets, securityGroup(s) and assignPublicIp can only be used in AwsVpc networking mode');
    }
}
/**
 * Force security group rules to be created in this stack.
 *
 * For every security group, if the scope and the group are in different stacks, return
 * a fake "imported" security group instead. This will behave as the original security group,
 * but new Ingress and Egress rule resources will be added in the current stack instead of the
 * other one.
 */
function securityGroupsInThisStack(scope, groups) {
    const thisStack = core_1.Stack.of(scope);
    let i = 1;
    return groups.map(group => {
        if (thisStack === core_1.Stack.of(group)) {
            return group;
        } // Simple case, just return the original one
        return ec2.SecurityGroup.fromSecurityGroupId(scope, `SecurityGroup${i++}`, group.securityGroupId, {
            allowAllOutbound: group.allowAllOutbound,
            mutable: true,
        });
    });
}
/**
 * The built-in container instance attributes
 */
class BuiltInAttributes {
}
exports.BuiltInAttributes = BuiltInAttributes;
/**
 * The id of the instance.
 */
BuiltInAttributes.INSTANCE_ID = 'instanceId';
/**
 * The AvailabilityZone where the instance is running in.
 */
BuiltInAttributes.AVAILABILITY_ZONE = 'attribute:ecs.availability-zone';
/**
 * The AMI id the instance is using.
 */
BuiltInAttributes.AMI_ID = 'attribute:ecs.ami-id';
/**
 * The EC2 instance type.
 */
BuiltInAttributes.INSTANCE_TYPE = 'attribute:ecs.instance-type';
/**
 * The operating system of the instance.
 *
 * Either 'linux' or 'windows'.
 */
BuiltInAttributes.OS_TYPE = 'attribute:ecs.os-type';
//# sourceMappingURL=data:application/json;base64,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