"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargatePlatformVersion = exports.FargateService = void 0;
const cdk = require("@aws-cdk/core");
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
/**
 * This creates a service using the Fargate launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class FargateService extends base_service_1.BaseService {
    /**
     * Imports from the specified service ARN.
     */
    static fromFargateServiceArn(scope, id, fargateServiceArn) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = fargateServiceArn;
                this.serviceName = cdk.Stack.of(scope).parseArn(fargateServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports from the specified service attrributes.
     */
    static fromFargateServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * Constructs a new instance of the FargateService class.
     */
    constructor(scope, id, props) {
        var _a;
        if (!props.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with Fargate');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            desiredCount: props.desiredCount !== undefined ? props.desiredCount : 1,
            launchType: base_service_1.LaunchType.FARGATE,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            platformVersion: props.platformVersion,
        }, props.taskDefinition);
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        if (!props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
}
exports.FargateService = FargateService;
/**
 * The platform version on which to run your service.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
 */
var FargatePlatformVersion;
(function (FargatePlatformVersion) {
    /**
     * The latest, recommended platform version.
     */
    FargatePlatformVersion["LATEST"] = "LATEST";
    /**
     * Version 1.4.0
     *
     * Supports EFS endpoints, CAP_SYS_PTRACE Linux capability,
     * network performance metrics in CloudWatch Container Insights,
     * consolidated 20 GB ephemeral volume.
     */
    FargatePlatformVersion["VERSION1_4"] = "1.4.0";
    /**
     * Version 1.3.0
     *
     * Supports secrets, task recycling.
     */
    FargatePlatformVersion["VERSION1_3"] = "1.3.0";
    /**
     * Version 1.2.0
     *
     * Supports private registries.
     */
    FargatePlatformVersion["VERSION1_2"] = "1.2.0";
    /**
     * Version 1.1.0
     *
     * Supports task metadata, health checks, service discovery.
     */
    FargatePlatformVersion["VERSION1_1"] = "1.1.0";
    /**
     * Initial release
     *
     * Based on Amazon Linux 2017.09.
     */
    FargatePlatformVersion["VERSION1_0"] = "1.0.0";
})(FargatePlatformVersion = exports.FargatePlatformVersion || (exports.FargatePlatformVersion = {}));
//# sourceMappingURL=data:application/json;base64,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