"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    /**
     * fluentbit
     */
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    /**
     * fluentd
     */
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    /**
     * s3
     */
    FirelensConfigFileType["S3"] = "s3";
    /**
     * fluentd
     */
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
//# sourceMappingURL=data:application/json;base64,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