"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryImage = void 0;
const core_1 = require("@aws-cdk/core");
const container_image_1 = require("../container-image");
/**
 * Regex pattern to check if it is an ECR image URL.
 *
 * @experimental
 */
const ECR_IMAGE_REGEX = /(^[a-zA-Z0-9][a-zA-Z0-9-_]*).dkr.ecr.([a-zA-Z0-9][a-zA-Z0-9-_]*).amazonaws.com(.cn)?\/.*/;
/**
 * An image hosted in a public or private repository. For images hosted in Amazon ECR, see
 * [EcrImage](https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html).
 */
class RepositoryImage extends container_image_1.ContainerImage {
    /**
     * Constructs a new instance of the RepositoryImage class.
     */
    constructor(imageName, props = {}) {
        super();
        this.imageName = imageName;
        this.props = props;
    }
    bind(scope, containerDefinition) {
        // name could be a Token - in that case, skip validation altogether
        if (!core_1.Token.isUnresolved(this.imageName) && ECR_IMAGE_REGEX.test(this.imageName)) {
            scope.node.addWarning("Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
        }
        if (this.props.credentials) {
            this.props.credentials.grantRead(containerDefinition.taskDefinition.obtainExecutionRole());
        }
        return {
            imageName: this.imageName,
            repositoryCredentials: this.props.credentials && {
                credentialsParameter: this.props.credentials.secretArn,
            },
        };
    }
}
exports.RepositoryImage = RepositoryImage;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVwb3NpdG9yeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlcG9zaXRvcnkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esd0NBQWlEO0FBRWpELHdEQUEwRTtBQUUxRTs7OztHQUlHO0FBQ0gsTUFBTSxlQUFlLEdBQUcsMEZBQTBGLENBQUM7QUFhbkg7OztHQUdHO0FBQ0gsTUFBYSxlQUFnQixTQUFRLGdDQUFjO0lBRWpEOztPQUVHO0lBQ0gsWUFBNkIsU0FBaUIsRUFBbUIsUUFBOEIsRUFBRTtRQUMvRixLQUFLLEVBQUUsQ0FBQztRQURtQixjQUFTLEdBQVQsU0FBUyxDQUFRO1FBQW1CLFVBQUssR0FBTCxLQUFLLENBQTJCO0lBRWpHLENBQUM7SUFFTSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxtQkFBd0M7UUFDcEUsbUVBQW1FO1FBQ25FLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUMvRSxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxxR0FBcUcsQ0FBQyxDQUFDO1NBQzlIO1FBRUQsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRTtZQUMxQixJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsY0FBYyxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztTQUM1RjtRQUVELE9BQU87WUFDTCxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDekIscUJBQXFCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLElBQUk7Z0JBQy9DLG9CQUFvQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVM7YUFDdkQ7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBMUJELDBDQTBCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHNlY3JldHNtYW5hZ2VyIGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb250YWluZXJEZWZpbml0aW9uIH0gZnJvbSAnLi4vY29udGFpbmVyLWRlZmluaXRpb24nO1xuaW1wb3J0IHsgQ29udGFpbmVySW1hZ2UsIENvbnRhaW5lckltYWdlQ29uZmlnIH0gZnJvbSAnLi4vY29udGFpbmVyLWltYWdlJztcblxuLyoqXG4gKiBSZWdleCBwYXR0ZXJuIHRvIGNoZWNrIGlmIGl0IGlzIGFuIEVDUiBpbWFnZSBVUkwuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5jb25zdCBFQ1JfSU1BR0VfUkVHRVggPSAvKF5bYS16QS1aMC05XVthLXpBLVowLTktX10qKS5ka3IuZWNyLihbYS16QS1aMC05XVthLXpBLVowLTktX10qKS5hbWF6b25hd3MuY29tKC5jbik/XFwvLiovO1xuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciBhbiBpbWFnZSBob3N0ZWQgaW4gYSBwdWJsaWMgb3IgcHJpdmF0ZSByZXBvc2l0b3J5LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlcG9zaXRvcnlJbWFnZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBzZWNyZXQgdG8gZXhwb3NlIHRvIHRoZSBjb250YWluZXIgdGhhdCBjb250YWlucyB0aGUgY3JlZGVudGlhbHMgZm9yIHRoZSBpbWFnZSByZXBvc2l0b3J5LlxuICAgKiBUaGUgc3VwcG9ydGVkIHZhbHVlIGlzIHRoZSBmdWxsIEFSTiBvZiBhbiBBV1MgU2VjcmV0cyBNYW5hZ2VyIHNlY3JldC5cbiAgICovXG4gIHJlYWRvbmx5IGNyZWRlbnRpYWxzPzogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcbn1cblxuLyoqXG4gKiBBbiBpbWFnZSBob3N0ZWQgaW4gYSBwdWJsaWMgb3IgcHJpdmF0ZSByZXBvc2l0b3J5LiBGb3IgaW1hZ2VzIGhvc3RlZCBpbiBBbWF6b24gRUNSLCBzZWVcbiAqIFtFY3JJbWFnZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUi9sYXRlc3QvdXNlcmd1aWRlL2ltYWdlcy5odG1sKS5cbiAqL1xuZXhwb3J0IGNsYXNzIFJlcG9zaXRvcnlJbWFnZSBleHRlbmRzIENvbnRhaW5lckltYWdlIHtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgUmVwb3NpdG9yeUltYWdlIGNsYXNzLlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBpbWFnZU5hbWU6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wczogUmVwb3NpdG9yeUltYWdlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogQ29uc3RydWN0LCBjb250YWluZXJEZWZpbml0aW9uOiBDb250YWluZXJEZWZpbml0aW9uKTogQ29udGFpbmVySW1hZ2VDb25maWcge1xuICAgIC8vIG5hbWUgY291bGQgYmUgYSBUb2tlbiAtIGluIHRoYXQgY2FzZSwgc2tpcCB2YWxpZGF0aW9uIGFsdG9nZXRoZXJcbiAgICBpZiAoIVRva2VuLmlzVW5yZXNvbHZlZCh0aGlzLmltYWdlTmFtZSkgJiYgRUNSX0lNQUdFX1JFR0VYLnRlc3QodGhpcy5pbWFnZU5hbWUpKSB7XG4gICAgICBzY29wZS5ub2RlLmFkZFdhcm5pbmcoXCJQcm9wZXIgcG9saWNpZXMgbmVlZCB0byBiZSBhdHRhY2hlZCBiZWZvcmUgcHVsbGluZyBmcm9tIEVDUiByZXBvc2l0b3J5LCBvciB1c2UgJ2Zyb21FY3JSZXBvc2l0b3J5Jy5cIik7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucHJvcHMuY3JlZGVudGlhbHMpIHtcbiAgICAgIHRoaXMucHJvcHMuY3JlZGVudGlhbHMuZ3JhbnRSZWFkKGNvbnRhaW5lckRlZmluaXRpb24udGFza0RlZmluaXRpb24ub2J0YWluRXhlY3V0aW9uUm9sZSgpKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgaW1hZ2VOYW1lOiB0aGlzLmltYWdlTmFtZSxcbiAgICAgIHJlcG9zaXRvcnlDcmVkZW50aWFsczogdGhpcy5wcm9wcy5jcmVkZW50aWFscyAmJiB7XG4gICAgICAgIGNyZWRlbnRpYWxzUGFyYW1ldGVyOiB0aGlzLnByb3BzLmNyZWRlbnRpYWxzLnNlY3JldEFybixcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuIl19