"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TmpfsMountOption = exports.DevicePermission = exports.Capability = exports.LinuxParameters = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Linux-specific options that are applied to the container.
 */
class LinuxParameters extends cdk.Construct {
    /**
     * Constructs a new instance of the LinuxParameters class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * Capabilities to be added
         */
        this.capAdd = new Array();
        /**
         * Capabilities to be dropped
         */
        this.capDrop = new Array();
        /**
         * Device mounts
         */
        this.devices = new Array();
        /**
         * TmpFs mounts
         */
        this.tmpfs = new Array();
        this.sharedMemorySize = props.sharedMemorySize;
        this.initProcessEnabled = props.initProcessEnabled;
    }
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    addCapabilities(...cap) {
        this.capAdd.push(...cap);
    }
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    dropCapabilities(...cap) {
        this.capDrop.push(...cap);
    }
    /**
     * Adds one or more host devices to a container.
     */
    addDevices(...device) {
        this.devices.push(...device);
    }
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     */
    addTmpfs(...tmpfs) {
        this.tmpfs.push(...tmpfs);
    }
    /**
     * Renders the Linux parameters to a CloudFormation object.
     */
    renderLinuxParameters() {
        return {
            initProcessEnabled: this.initProcessEnabled,
            sharedMemorySize: this.sharedMemorySize,
            capabilities: {
                add: cdk.Lazy.listValue({ produce: () => this.capAdd }, { omitEmpty: true }),
                drop: cdk.Lazy.listValue({ produce: () => this.capDrop }, { omitEmpty: true }),
            },
            devices: cdk.Lazy.anyValue({ produce: () => this.devices.map(renderDevice) }, { omitEmptyArray: true }),
            tmpfs: cdk.Lazy.anyValue({ produce: () => this.tmpfs.map(renderTmpfs) }, { omitEmptyArray: true }),
        };
    }
}
exports.LinuxParameters = LinuxParameters;
function renderDevice(device) {
    return {
        containerPath: device.containerPath,
        hostPath: device.hostPath,
        permissions: device.permissions,
    };
}
function renderTmpfs(tmpfs) {
    return {
        containerPath: tmpfs.containerPath,
        size: tmpfs.size,
        mountOptions: tmpfs.mountOptions,
    };
}
/**
 * A Linux capability
 */
var Capability;
(function (Capability) {
    Capability["ALL"] = "ALL";
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    Capability["CHOWN"] = "CHOWN";
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    Capability["FOWNER"] = "FOWNER";
    Capability["FSETID"] = "FSETID";
    Capability["IPC_LOCK"] = "IPC_LOCK";
    Capability["IPC_OWNER"] = "IPC_OWNER";
    Capability["KILL"] = "KILL";
    Capability["LEASE"] = "LEASE";
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    Capability["MKNOD"] = "MKNOD";
    Capability["NET_ADMIN"] = "NET_ADMIN";
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    Capability["NET_RAW"] = "NET_RAW";
    Capability["SETFCAP"] = "SETFCAP";
    Capability["SETGID"] = "SETGID";
    Capability["SETPCAP"] = "SETPCAP";
    Capability["SETUID"] = "SETUID";
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    Capability["SYS_BOOT"] = "SYS_BOOT";
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    Capability["SYS_MODULE"] = "SYS_MODULE";
    Capability["SYS_NICE"] = "SYS_NICE";
    Capability["SYS_PACCT"] = "SYS_PACCT";
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    Capability["SYS_TIME"] = "SYS_TIME";
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    Capability["SYSLOG"] = "SYSLOG";
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
/**
 * Permissions for device access
 */
var DevicePermission;
(function (DevicePermission) {
    /**
     * Read
     */
    DevicePermission["READ"] = "read";
    /**
     * Write
     */
    DevicePermission["WRITE"] = "write";
    /**
     * Make a node
     */
    DevicePermission["MKNOD"] = "mknod";
})(DevicePermission = exports.DevicePermission || (exports.DevicePermission = {}));
/**
 * The supported options for a tmpfs mount for a container.
 */
var TmpfsMountOption;
(function (TmpfsMountOption) {
    TmpfsMountOption["DEFAULTS"] = "defaults";
    TmpfsMountOption["RO"] = "ro";
    TmpfsMountOption["RW"] = "rw";
    TmpfsMountOption["SUID"] = "suid";
    TmpfsMountOption["NOSUID"] = "nosuid";
    TmpfsMountOption["DEV"] = "dev";
    TmpfsMountOption["NODEV"] = "nodev";
    TmpfsMountOption["EXEC"] = "exec";
    TmpfsMountOption["NOEXEC"] = "noexec";
    TmpfsMountOption["SYNC"] = "sync";
    TmpfsMountOption["ASYNC"] = "async";
    TmpfsMountOption["DIRSYNC"] = "dirsync";
    TmpfsMountOption["REMOUNT"] = "remount";
    TmpfsMountOption["MAND"] = "mand";
    TmpfsMountOption["NOMAND"] = "nomand";
    TmpfsMountOption["ATIME"] = "atime";
    TmpfsMountOption["NOATIME"] = "noatime";
    TmpfsMountOption["DIRATIME"] = "diratime";
    TmpfsMountOption["NODIRATIME"] = "nodiratime";
    TmpfsMountOption["BIND"] = "bind";
    TmpfsMountOption["RBIND"] = "rbind";
    TmpfsMountOption["UNBINDABLE"] = "unbindable";
    TmpfsMountOption["RUNBINDABLE"] = "runbindable";
    TmpfsMountOption["PRIVATE"] = "private";
    TmpfsMountOption["RPRIVATE"] = "rprivate";
    TmpfsMountOption["SHARED"] = "shared";
    TmpfsMountOption["RSHARED"] = "rshared";
    TmpfsMountOption["SLAVE"] = "slave";
    TmpfsMountOption["RSLAVE"] = "rslave";
    TmpfsMountOption["RELATIME"] = "relatime";
    TmpfsMountOption["NORELATIME"] = "norelatime";
    TmpfsMountOption["STRICTATIME"] = "strictatime";
    TmpfsMountOption["NOSTRICTATIME"] = "nostrictatime";
    TmpfsMountOption["MODE"] = "mode";
    TmpfsMountOption["UID"] = "uid";
    TmpfsMountOption["GID"] = "gid";
    TmpfsMountOption["NR_INODES"] = "nr_inodes";
    TmpfsMountOption["NR_BLOCKS"] = "nr_blocks";
    TmpfsMountOption["MPOL"] = "mpol";
})(TmpfsMountOption = exports.TmpfsMountOption || (exports.TmpfsMountOption = {}));
//# sourceMappingURL=data:application/json;base64,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